package com.atlassian.crowd.model.group;

import com.google.common.base.MoreObjects;

import javax.annotation.Nonnull;
import java.util.Objects;

public abstract class BaseImmutableGroup implements Group {
    private final long directoryId;
    private final String name;
    private final GroupType type;
    private final boolean active;
    private final String description;
    private final String externalId;

    protected BaseImmutableGroup(final Builder builder) {
        this.directoryId = builder.directoryId;
        this.name = builder.name;
        this.type = builder.type;
        this.active = builder.active;
        this.description = builder.description;
        this.externalId = builder.externalId;
    }

    abstract public BaseImmutableGroup withName(String name);

    @Override
    public int compareTo(@Nonnull Group other) {
        return GroupComparator.compareTo(this, other);
    }

    @Override
    public long getDirectoryId() {
        return directoryId;
    }

    @Override
    public String getName() {
        return name;
    }

    @Override
    public GroupType getType() {
        return type;
    }

    @Override
    public boolean isActive() {
        return active;
    }

    @Override
    public String getDescription() {
        return description;
    }

    @Override
    public String getExternalId() {
        return externalId;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;
        BaseImmutableGroup that = (BaseImmutableGroup) o;
        return directoryId == that.directoryId &&
                active == that.active &&
                Objects.equals(name, that.name) &&
                type == that.type &&
                Objects.equals(description, that.description);
    }

    @Override
    public int hashCode() {
        return Objects.hash(directoryId, name, type, active, description);
    }

    protected static abstract class Builder<T extends Builder> {
        private long directoryId;
        private String name;
        private GroupType type = GroupType.GROUP;
        private boolean active = true;
        private String description = "";
        private String externalId;

        protected Builder(Group group) {
            Objects.requireNonNull(group, "group");
            setDirectoryId(group.getDirectoryId());
            setName(group.getName());
            setDescription(group.getDescription());
            setType(group.getType());
            setActive(group.isActive());
            setDescription(group.getDescription());
            setExternalId(group.getExternalId());
        }

        public Builder(String name) {
            this(-1L, name);
        }

        public Builder(long directoryId, String name) {
            this.directoryId = directoryId;
            this.name = name;
        }

        public T setDirectoryId(long directoryId) {
            this.directoryId = directoryId;
            return (T) this;
        }

        public T setName(String name) {
            this.name = name;
            return (T) this;
        }

        public T setType(GroupType type) {
            this.type = type;
            return (T) this;
        }

        public T setActive(boolean active) {
            this.active = active;
            return (T) this;
        }

        public T setDescription(String description) {
            this.description = description;
            return (T) this;
        }

        public T setExternalId(String externalId) {
            this.externalId = externalId;
            return (T) this;
        }

        public abstract Group build();
    }

    @Override
    public String toString() {
        return MoreObjects.toStringHelper(this)
                .add("directoryId", directoryId)
                .add("name", name)
                .add("type", type)
                .add("active", active)
                .add("description", description)
                .add("externalId", externalId)
                .toString();
    }
}
