package com.atlassian.crowd.util;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.function.Function;
import java.util.function.Supplier;

/**
 * A batch mutation operation result representing the collection of entities
 * that were successfully processed and the collection of entities that failed
 * processing.
 */
public class BatchResult<T> {
    private final List<T> successfulEntities;
    private final List<T> failedEntities;

    public static <T, V, K extends BatchResult<V>> K transform(BatchResult<T> batchResult, Function<T, V> transformer, Supplier<K> supplier) {
        K transformedBatchResult = supplier.get();
        batchResult.getSuccessfulEntities().forEach((entity) -> transformedBatchResult.addSuccess(transformer.apply(entity)));
        batchResult.getFailedEntities().forEach((entity) -> transformedBatchResult.addFailure(transformer.apply(entity)));
        return transformedBatchResult;
    }

    public static <T, V> BatchResult<V> transform(BatchResult<T> batchResult, Function<T, V> transformer) {
        return transform(batchResult, transformer, () -> new BatchResult<>(batchResult.getTotalAttempted()));
    }

    public BatchResult(int totalEntities) {
        successfulEntities = new ArrayList<>(totalEntities);
        failedEntities = new ArrayList<>();
    }

    public void addSuccess(T entity) {
        successfulEntities.add(entity);
    }

    public void addSuccesses(Collection<? extends T> entities) {
        successfulEntities.addAll(entities);
    }

    public void addFailure(T entity) {
        failedEntities.add(entity);
    }

    public void addFailures(Collection<? extends T> entities) {
        failedEntities.addAll(entities);
    }

    public boolean hasFailures() {
        return !failedEntities.isEmpty();
    }

    public int getTotalAttempted() {
        return successfulEntities.size() + failedEntities.size();
    }

    public List<T> getSuccessfulEntities() {
        return successfulEntities;
    }

    public List<T> getFailedEntities() {
        return failedEntities;
    }

    public int getTotalSuccessful() {
        return successfulEntities.size();
    }
}
