package com.atlassian.crowd.manager.sso;

import com.atlassian.crowd.exception.OperationFailedException;
import com.atlassian.crowd.model.application.Application;
import com.atlassian.crowd.model.sso.ApplicationSamlConfiguration;
import com.atlassian.crowd.model.sso.BaseApplicationSamlConfiguration;

import java.io.InputStream;
import java.util.Optional;

/**
 * A service for managing an application's SAML SSO configuration
 * @since 3.4.0
 */
public interface ApplicationSamlConfigurationService {

    /**
     * Returns the SAML SSO configuration for the specified application
     * @param application the application for which the configuration should be fetched
     * @return an optional containing the configuration if present or an empty one if not
     */
    Optional<ApplicationSamlConfiguration> loadConfigurationForApplication(Application application);

    /**
     * Creates or updates SAML SSO configuration for an application
     * @param configuration the configuration to save or update
     */
    void storeApplicationConfiguration(ApplicationSamlConfiguration configuration) throws InvalidApplicationSamlConfigurationException, OperationFailedException;

    /**
     * Returns the SAML SSO configuration for the specified assertion consumer url and audience. At least one of the
     * parameters have to be specified to get non-empty result.
     * @param assertionConsumerServiceURL SSO application assertion consumer url
     * @param audience SSO application audience
     * @return an optional with the corresponding configuration, if any
     */
    Optional<ApplicationSamlConfiguration> findByAssertionConsumerAndAudience(String assertionConsumerServiceURL, String audience);

    /**
     * Parses SAML SSO configuration for an application based on the provided XML metadata.
     * @param metadataXml stream with XML metadata, will be closed by this method
     * @return parsed configuration
     * @throws IllegalArgumentException when input stream is not valid XML metadata
     * @see <a href="https://www.oasis-open.org/committees/download.php/22387/sstc-saml-metadata-errata-2.0-wd-03-diff.pdf">Metadata for SAML</a>
     */
    BaseApplicationSamlConfiguration parseApplicationMetadata(InputStream metadataXml);
}
