package com.atlassian.crowd.manager.directory;

import com.atlassian.crowd.exception.DirectoryNotFoundException;
import com.atlassian.crowd.exception.FeatureInaccessibleException;
import com.atlassian.crowd.exception.GroupNotFoundException;
import com.atlassian.crowd.exception.OperationFailedException;
import com.atlassian.crowd.exception.UserNotFoundException;
import com.atlassian.crowd.manager.permission.DirectoryGroup;
import com.atlassian.crowd.model.user.User;
import com.atlassian.crowd.util.BatchResult;
import com.atlassian.crowd.util.BatchResultEntityWithCause;

import java.util.Collection;
import java.util.List;

/**
 * Represents a component which manages memberships for groups as per available permission
 *
 * @since 3.3.0
 */
public interface MembershipService {
    /**
     * Adds multiple users to single supplied group
     *
     * @param directoryId ID of the directory, where supplied users and group are placed
     * @param usernames   List of usernames to add to group
     * @param groupName   Name of the target group
     * @return Failures occurred while adding. Does not count users, which are already assigned to the group
     * @throws GroupNotFoundException if the group cannot be found
     * @throws DirectoryNotFoundException if the directory cannot be found    * @throws DirectoryPermissionException if
     * @throws DirectoryPermissionException if the directory is not allowed to perform operations of type {@link com.atlassian.crowd.embedded.api.OperationType#UPDATE_GROUP}.
     * @throws OperationFailedException if underlying directory implementation failed to execute the operation.
     * @throws FeatureInaccessibleException If the feature is not available under the current license
     */
    BatchResult<BatchResultEntityWithCause> addUsersToGroup(long directoryId, List<String> usernames, String groupName)
            throws GroupNotFoundException, DirectoryNotFoundException, DirectoryPermissionException, OperationFailedException,
                   FeatureInaccessibleException;

    /**
     * Adds single supplied user to multiple groups
     *
     * @param directoryId ID of the directory, where supplied user and groups are placed
     * @param username    Name of the user to add
     * @param groupNames  List of target group names
     * @return Failures occurred while adding. Does not count users, which are already assigned to the group
     * @throws UserNotFoundException if the user cannot be found
     * @throws DirectoryNotFoundException if the directory cannot be found
     * @throws OperationFailedException if underlying directory implementation failed to execute the operation.
     * @throws FeatureInaccessibleException If the feature is not available under the current license
     */
    BatchResult<BatchResultEntityWithCause> addUserToGroups(long directoryId, String username, List<String> groupNames)
            throws UserNotFoundException, DirectoryNotFoundException, OperationFailedException, FeatureInaccessibleException;

    /**
     * Removes multiple users from the supplied group
     *
     * @param directoryId ID of the directory, where the supplied users and group are placed
     * @param usernames   List of usernames to remove from group
     * @param groupName   Name of the target group
     * @return Failures occurred while removing users from the group.
     * Does not count users, which are not assigned to the group
     * @throws GroupNotFoundException if the group cannot be found
     * @throws DirectoryNotFoundException if the directory cannot be found
     * @throws OperationFailedException if underlying directory implementation failed to execute the operation.\
     * @throws FeatureInaccessibleException If the feature is not available under the current license
     */
    BatchResult<BatchResultEntityWithCause> removeUsersFromGroup(long directoryId, List<String> usernames, String groupName)
            throws GroupNotFoundException, DirectoryNotFoundException, OperationFailedException, FeatureInaccessibleException;

    /**
     * Removes single supplied user from multiple groups
     *
     * @param directoryId ID of directory, where supplied user and groups are placed
     * @param username    Name of the user to remove from the group
     * @param groupNames  List of the target group names
     * @return Failures occurred while removing user from groups.
     * Does not count groups, which are not associated with user
     * @throws UserNotFoundException if the user cannot be found
     * @throws DirectoryNotFoundException if the directory cannot be found
     * @throws OperationFailedException if underlying directory implementation failed to execute the operation.
     * @throws FeatureInaccessibleException If the feature is not available under the current license
     */
    BatchResult<BatchResultEntityWithCause> removeUserFromGroups(long directoryId, String username, List<String> groupNames)
            throws UserNotFoundException, DirectoryNotFoundException, OperationFailedException, FeatureInaccessibleException;

    /**
     * Returns details of specific group
     *
     * @param directoryId   ID of directory, where group is placed
     * @param groupName     Name of group to look for
     * @throws DirectoryNotFoundException if the directory cannot be found
     * @throws GroupNotFoundException if the group cannot be found
     * @throws OperationFailedException if underlying directory implementation failed to execute the operation.
     * @throws FeatureInaccessibleException If the feature is not available under the current license
     */
    DirectoryGroup getGroupDetails(long directoryId, String groupName)
            throws DirectoryNotFoundException, GroupNotFoundException, OperationFailedException, FeatureInaccessibleException;

    /**
     * Returns direct members of a group (excluding nested members)
     *
     * @param directoryId   ID of directory, where group is placed
     * @param groupName     The group name to look for users
     * @param startIndex    Starting index of the results
     * @param maxResults    Maximum number of results returned
     * @throws DirectoryNotFoundException if the directory cannot be found
     * @throws OperationFailedException if underlying directory implementation failed to execute the operation.
     * @throws GroupNotFoundException if the group cannot be found
     * @throws FeatureInaccessibleException If the feature is not available under the current license
     */
    Collection<User> getDirectUsersFromGroup(long directoryId, String groupName, int startIndex, int maxResults)
            throws DirectoryNotFoundException, OperationFailedException, GroupNotFoundException, FeatureInaccessibleException;
}
