package com.atlassian.crowd.audit;

import com.google.common.base.MoreObjects;

import javax.annotation.Nullable;
import java.util.Objects;

/**
 * Concrete implementation of an object affected by an auditable event
 */
public class ImmutableAuditLogEntity implements AuditLogEntity {

    private final AuditLogEntityType entityType;
    private final Long entityId;
    private final String entityName;
    private final boolean primary;

    public ImmutableAuditLogEntity(Builder builder) {
        this.entityType = builder.entityType;
        this.entityId = builder.entityId;
        this.entityName = builder.entityName;
        this.primary = builder.primary;
    }

    @Nullable
    @Override
    public AuditLogEntityType getEntityType() {
        return entityType;
    }

    @Nullable
    @Override
    public Long getEntityId() {
        return entityId;
    }

    @Nullable
    @Override
    public String getEntityName() {
        return entityName;
    }

    @Override
    public boolean isPrimary() {
        return primary;
    }

    public static ImmutableAuditLogEntity from(AuditLogEntity original) {
        return original instanceof ImmutableAuditLogEntity ? (ImmutableAuditLogEntity) original : new Builder(original).build();
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;
        ImmutableAuditLogEntity that = (ImmutableAuditLogEntity) o;
        return entityType == that.entityType &&
                Objects.equals(entityId, that.entityId) &&
                Objects.equals(entityName, that.entityName) &&
                Objects.equals(primary, that.primary);
    }

    @Override
    public String toString() {
        return MoreObjects.toStringHelper(this)
                .add("entityType", entityType)
                .add("entityId", entityId)
                .add("entityName", entityName)
                .add("primary", primary)
                .toString();
    }

    @Override
    public int hashCode() {
        return Objects.hash(entityType, entityId, entityName, primary);
    }

    public static class Builder {
        private AuditLogEntityType entityType;
        private Long entityId;
        private String entityName;
        private boolean primary;

        public Builder() {}

        public Builder(AuditLogEntity entity) {
            this.entityType = entity.getEntityType();
            this.entityId = entity.getEntityId();
            this.entityName = entity.getEntityName();
            this.primary = entity.isPrimary();
        }

        public Builder setEntityType(AuditLogEntityType entityType) {
            this.entityType = entityType;
            return this;
        }

        public Builder setEntityId(Long entityId) {
            this.entityId = entityId;
            return this;
        }

        public Builder setEntityName(String entityName) {
            this.entityName = entityName;
            return this;
        }

        public Builder setPrimary() {
            this.primary = true;
            return this;
        }

        public ImmutableAuditLogEntity build() {
            return new ImmutableAuditLogEntity(this);
        }
    }
}
