package com.atlassian.crowd.audit;

import javax.annotation.Nullable;
import java.time.Instant;
import java.util.Collection;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;

/**
 * Represents a set of changes that was saved in an audit log. Contains a changeset id and at least one entry
 */
public interface AuditLogChangeset {
    /**
     * @return The changeset identifier
     */
    Long getId();

    /**
     * @return the creation date of the audit log entry
     */
    Instant getTimestampInstant();

    /**
     * @since 3.2.0
     * @return the information about the author of the audit log entry (exact semantic depends on the author type)
     */
    AuditLogAuthor getAuthor();

    /**
     * @deprecated Use {@link #getAuthor()}#} instead. Since 3.2.0.
     * @return the type of the author of the audit log entry
     */
    @Deprecated
    AuditLogAuthorType getAuthorType();

    /**
     * @deprecated Use {@link #getAuthor()}#} instead. Since 3.2.0.
     * @return the identifier of the author of the audit log entry (exact semantic depends on the author type)
     */
    @Deprecated
    @Nullable
    Long getAuthorId();

    /**
     * Returns the name of the author of the audit log entry. This is particularly useful when the author does not
     * exist anymore
     *
     * @deprecated Use {@link #getAuthor()}#} instead. Since 3.2.0.
     * @return the name of the author of the audit log entry
     */
    @Deprecated
    @Nullable
    String getAuthorName();

    /**
     * @return information about the type of the event represented by this audit log entry
     */
    AuditLogEventType getEventType();

    /**
     * @since 3.2.0
     * @return
     * <ul>
     *     <li>If the event did not have affected entities an empty optional will be returned</li>
     *     <li>If the event affected only one entity then the object will be returned</li>
     *     <li>
     *         If the event affected multiple entities then the entity marked as primary will be returned.
     *         If there are multiple entities marked as primary, then there are no guarantees which one of them will
     *         be returned.
     *     </li>
     * </ul>
     */
    default Optional<AuditLogEntity> getEntity() {
        if (getEntities().isEmpty()) {
            return Optional.empty();
        } else {
            final Optional<? extends AuditLogEntity> primaryObject = getEntities().stream()
                    .filter(AuditLogEntity::isPrimary)
                    .findFirst();
            return primaryObject.<Optional<AuditLogEntity>>map(Optional::of)
                    .orElseGet(() -> Optional.of(getEntities().iterator().next()));
        }
    }

    /**
     * Will return additional objects affected by this audit event. Additional objects are all objects that are marked
     * as non-primary.
     * @return additional objects affected by this audit event
     */
    default List<AuditLogEntity> getAdditionalEntities() {
        return getAdditionalEntities().stream()
                .filter(ao -> !ao.isPrimary())
                .collect(Collectors.toList());
    }

    /**
     * @return the type of the object that was modified in the event represented by this audit log entry
     *
     * @deprecated use the {@link AuditLogChangeset#getEntities()} instead
     */
    @Nullable
    @Deprecated
    default AuditLogEntityType getEntityType() {
        final Optional<AuditLogEntity> primaryObject = getEntity();
        if (primaryObject.isPresent()) {
            return primaryObject.get().getEntityType();
        } else {
            return null;
        }
    }

    /**
     * @return the identifier of the object that was modified in the event represented by this audit log entry
     * (exact semantic depends on the entity type)
     * @deprecated use the {@link AuditLogChangeset#getEntities()} instead
     */
    @Nullable
    @Deprecated
    default Long getEntityId() {
        final Optional<AuditLogEntity> primaryObject = getEntity();
        if (primaryObject.isPresent()) {
            return primaryObject.get().getEntityId();
        } else {
            return null;
        }
    }

    /**
     * Returns the name of the object that was modified in the event represented by this audit log entry. This is
     * particularly useful when the object does not exist anymore
     *
     * @return the name of the affected object at the time the event occurred
     * @deprecated use the {@link AuditLogChangeset#getEntities()} instead
     */
    @Nullable
    @Deprecated
    default String getEntityName() {
        final Optional<AuditLogEntity> primaryObject = getEntity();
        if (primaryObject.isPresent()) {
            return primaryObject.get().getEntityName();
        } else {
            return null;
        }
    }

    /**
     * @return the IP address of the entity which performed the auditable action
     */
    @Nullable
    String getIpAddress();

    /**
     * @return custom message providing additional detail about the auditable event
     */
    @Nullable
    String getEventMessage();

    /**
     * @return the source of the auditable event (whether it was manual or changed by an automated process for example)
     */
    AuditLogEventSource getSource();

    /**
     * @return the entries contained in the changeset
     */
    Collection<? extends AuditLogEntry> getEntries();

    /**
     * @return the objects that were affected by the event reflected by the changeset
     */
    Collection<? extends AuditLogEntity> getEntities();
}
