package com.atlassian.crowd.directory;

import java.util.Arrays;

import javax.annotation.Nonnull;

import com.atlassian.crowd.embedded.api.Attributes;

/**
 * Constants representing synchronisable directory properties.
 */
public class SynchronisableDirectoryProperties {
    /**
     * Property key for the current synchronisation's start time in milliseconds from epoch.
     */
    public static final String CURRENT_START_SYNC_TIME = "com.atlassian.crowd.directory.sync.currentstartsynctime";

    /**
     * Property key for the last synchronisation's start time in milliseconds from epoch.
     */
    public static final String LAST_START_SYNC_TIME = "com.atlassian.crowd.directory.sync.laststartsynctime";

    /**
     * Property key for the last synchronisation's duration in milliseconds.
     */
    public static final String LAST_SYNC_DURATION_MS = "com.atlassian.crowd.directory.sync.lastdurationms";

    /**
     * Property key for the interval in seconds when the local cache should be synchronised with the remote directory.
     */
    public static final String CACHE_SYNCHRONISE_INTERVAL = "directory.cache.synchronise.interval";

    /**
     * Property key for the synchronisation status.
     * <p>
     * This property should be unset before the first synchronisation has
     * started after directory creation or configuration change.
     */
    public static final String IS_SYNCHRONISING = "com.atlassian.crowd.directory.sync.issynchronising";

    /**
     * Property key for enabling incremental sync
     */
    public static final String INCREMENTAL_SYNC_ENABLED = "crowd.sync.incremental.enabled";

    /**
     * Property key for syncing a subset of users for a {@link SynchronisableDirectory}, as long as
     * {@link com.atlassian.crowd.manager.directory.SynchronisationMode#FULL} is the <em>only</em> mode being used.
     *
     * The value of the attribute with this key must be CQL and is used to filter remote users: only users which match
     * the CQL will be synchronised to the directory.
     *
     * Setting this property on a directory of type other than {@link SynchronisableDirectory} should have no effect.
     *
     * Setting this property when {@link com.atlassian.crowd.manager.directory.SynchronisationMode#FULL} is not the
     * <em>only</em> mode being used will result in undefined behaviour.
     *
     * Setting this property as a Java system property is equivalent to setting it on every synchronisable directory. If
     * set as both a Java system property and on the directory, then the Java system property will take precedence.
     */
    public static final String SYNC_USER_FILTER_CQL = "com.atlassian.crowd.sync.user.filter.cql";

    /**
     * Property key for controlling group membership sync after successful user auth.
     *
     * This operation can slow down user auth check significantly if the directory contains lots of groups.
     * In that case, Crowd admin might want to disable this feature and rely on scheduled or manual directory sync to
     * update the group membership details for the affected users.
     *
     * Possible values are managed by {@link SyncGroupMembershipsAfterAuth}
     */
    public static final String SYNC_GROUP_MEMBERSHIP_AFTER_SUCCESSFUL_USER_AUTH_ENABLED = "crowd.sync.group.membership.after.successful.user.auth.enabled";

    /**
     * Allows configuring whether user groups should be synchronised when the user authenticates
     */
    public enum SyncGroupMembershipsAfterAuth {
        /**
         * Never synchronize group memberships during authentication - this was the default
         * before this feature was introduced in 2.7 (CWD-3429)
         */
        NEVER("false"),
        /**
         * Always sync group memberships during authentication - this was the default since 2.7, until 2.10
         * There is a potential performance impact on querying the groups on login, even if the list of memberships hasn't changed.
         */
        ALWAYS("true"),

        /**
         * Sync when the user has been created as a part of the authentication process, - this is the new default since 2.10 (CWD-4197).
         * This is a compromise between the convenience of being able to log in (and being in the right groups) without waiting for a sync,
         * and allows for faster subsequent authentication.
         */
        WHEN_AUTHENTICATION_CREATED_THE_USER("only_when_first_created");

        private final String value;

        /**
         * This should be the default for new directories
         */
        public final static SyncGroupMembershipsAfterAuth DEFAULT = WHEN_AUTHENTICATION_CREATED_THE_USER;

        /**
         * This was the behavior for pre-2.10 directories. Keep it for directories that don't have an explicit setting
         */
        private final static SyncGroupMembershipsAfterAuth LEGACY_DEFAULT = ALWAYS;

        SyncGroupMembershipsAfterAuth(String value) {
            this.value = value;
        }

        public static SyncGroupMembershipsAfterAuth forDirectory(@Nonnull Attributes directoryWithAttributes) {
            return forValue(directoryWithAttributes.getValue(SYNC_GROUP_MEMBERSHIP_AFTER_SUCCESSFUL_USER_AUTH_ENABLED));
        }

        public static SyncGroupMembershipsAfterAuth forValue(String attributeValue) {
            return Arrays.stream(SyncGroupMembershipsAfterAuth.values())
                    .filter(item -> item.value.equals(attributeValue))
                    .findFirst()
                    .orElse(LEGACY_DEFAULT);
        }

        public String getValue() {
            return value;
        }
    }

    private SynchronisableDirectoryProperties() {
    }
}
