var RY = RY || {};

RY.PageView = Backbone.View.extend({
    tagName: "li",
    className: "ry-page",
    template: RY.Templates.pageItem,

    initialize: function() {
        _.bindAll(this, 'hide', 'show', 'render');
    },

    hide: function() {
        this.$el.hide().removeClass("shown");
    },

    show: function() {
        this.$el.show().addClass("shown");
    },

    render: function() {
        var json = this.model.toJSON();
        json.href = this.model.href();

        this.$el.html(this.template(json));
        return this;
    }
});

RY.PageGroupView = Backbone.View.extend({
    className: "group",
    template: RY.Templates.pageGroup,

    initialize: function() {
        _.bindAll(this, 'hide', 'hideAll', 'show', 'showBorder', 'showPages', 'add', 'render');
        this.title = this.options.title;
        this.modelViews = {};
    },

    hide: function() {
        this.$el.hide();
    },

    hideAll: function() {
        this.$el.removeClass("border").hide();

        // hide views
        _.invoke(_.values(this.modelViews), "hide");
    },

    show: function() {
        this.$el.show();
    },

    showBorder: function() {
        this.$el.addClass("border");
    },

    showPages: function(pages) {
        var that = this;
        var hasPage = false;
        _.each(pages, function(page) {
            var view = that.modelViews[page.cid];
            if (view) {
                hasPage = true;
                view.show();
            }
        });

        if (hasPage) {
            this.show();
        }

        return hasPage;
    },

    add: function(page) {
        var view = new RY.PageView({model: page});
        this.modelViews[page.cid] = view;

        this.$list.append(view.render().el);
    },

    render: function() {
        this.$el.html(this.template({
            title: this.title
        }));
        this.$list = this.$("ul");

        return this;
    }
});

RY.PageNavigator = function($pageContainer, $scrollContainer, navigationEvents) {
    var activeItem = null;

    function getPageItems() {
        return $pageContainer.find("li.shown");
    }

    function selectItem(next) {
        pageItems = getPageItems();
        var index = pageItems.index(pageItems.filter(".highlight"));

        // unselect old one
        if (activeItem) {
            activeItem.removeClass("highlight");
        }

        next += index;

        // keep in bounds
        if (next < 0) {
            next = pageItems.length - 1;
        }

        if (next >= pageItems.length) {
            next = 0;
        }

        // select
        activeItem = pageItems.eq(next);
        activeItem.addClass("highlight");
    }

    /**
     * Keeps the active item visible by scrolling the container.
     */
    function scrollIfNecessary() {
        if (!activeItem.length) {
            return;
        }
        
        var scrollContainer = $scrollContainer;
        var content = scrollContainer.children();
        var scHeight = scrollContainer.height();
        var elHeight = activeItem.outerHeight(true);

        // scroll element into view at the top or at the bottom
        var position = activeItem.position().top;
        if (position < 0) {
            // scroll to top when last item flips over to first
            if (getPageItems().index(activeItem) === 0) {
                scrollContainer.scrollTop(0);
                return;
            }

            // scroll up
            scrollContainer.scrollTop(activeItem.offset().top - content.offset().top);

        } else if (position > scHeight) {
            // scroll down
            scrollContainer.scrollTop(activeItem.offset().top - content.offset().top - scHeight + elHeight);
        }
    }

    navigationEvents.on("select", function() {
        if (activeItem && activeItem.is(":visible")) {
            RY.util.analytics.trackPageOpen();
            var href = activeItem.find(".page-title span").attr('href');
            window.location = href;
        }
    }, this);

    navigationEvents.on("previous", function() {
        selectItem(-1);
        scrollIfNecessary();
    }, this);


    navigationEvents.on("next", function() {
        selectItem(1);
        scrollIfNecessary();
    }, this);
};

RY.PageCollectionView = Backbone.View.extend({
    template: RY.Templates.pageCollection,

    events: {
        "click .page-title a" : RY.util.analytics.trackPageOpen
    },

    initialize: function() {
        _.bindAll(this, 'checkEmpty', 'filter', '_groupForPage', 'addOne', 'showEmptyMessage', 
            'clearEmptyMessage', 'addAll', 'render');
        this.modelViews = {};
        this.collection.on("reset", this.addAll, this);
        this.collection.on("add", this.addOne, this);
        this.collection.on("filter", this.filter, this);
    },

    checkEmpty: function(pages, query) {
        var noHistory = this.collection.isEmpty();
        var noResults = pages.length === 0;
        if (noHistory || noResults) {
            var message;
            if (noHistory) {
                message = AJS.I18n.getText("com.atlassian.confluence.plugins.recently.viewed.no_history");
            } else {
                var searchUrl = AJS.contextPath() + "/dosearchsite.action?queryString=" + encodeURIComponent(query);
                message = AJS.I18n.getText("com.atlassian.confluence.plugins.recently.viewed.no_results") + " " +
                    AJS.I18n.getText("com.atlassian.confluence.plugins.recently.viewed.no_results.search", searchUrl);
            }
            this.showEmptyMessage(message);
        } else {
            this.clearEmptyMessage();
        }
    },

    /**
     * Filters the list view to show only the page models from the pages array.
     * @param pages
     * @param query
     */
    filter: function(pages, query) {
        query = query || "";
        
        this.checkEmpty(pages, query);

        var groups = [this.$today, this.$yesterday, this.$older];

        // hide all groups and borders
        _.invoke(groups, "hideAll");

        // show pages in each group
        var groupsWithPages = [];
        _.each(groups, function(group) {
            var hasPage = group.showPages(pages);
            if (hasPage) {
                groupsWithPages.push(group);
            }
        });

        // add border to all shown groups except for the last
        if (groupsWithPages.length > 1) {
            groupsWithPages.pop();
            _.invoke(groupsWithPages, "showBorder");
        }
    },

    _groupForPage: function(page) {
        var days = page.daysSinceLastSeen();

        if (days === 0) {
            return this.$today;
        } else if (days === 1) {
            return this.$yesterday;
        } else {
            return this.$older;
        }
    },

    addOne: function(page) {
        var $toInsert = this._groupForPage(page);
        $toInsert.add(page);
    },

    showEmptyMessage: function(message) {
        this.$(".empty").html(AJS.$("<p>").html(message));
    },

    clearEmptyMessage: function() {
        this.$(".empty").html("");
    },

    addAll: function() {
        this.collection.each(this.addOne);
    },

    render: function() {
        this.$el.html(this.template());

        this.$today = new RY.PageGroupView({
            title: AJS.I18n.getText("com.atlassian.confluence.plugins.recently.viewed.date.today")
        });

        this.$yesterday = new RY.PageGroupView({
            title: AJS.I18n.getText("com.atlassian.confluence.plugins.recently.viewed.date.yesterday")
        });

        this.$older = new RY.PageGroupView({
            title: AJS.I18n.getText("com.atlassian.confluence.plugins.recently.viewed.date.older")
        });

        var $groups = this.$(".groups");
        $groups.append(this.$today.render().el);
        $groups.append(this.$yesterday.render().el);
        $groups.append(this.$older.render().el);

        _.invoke([this.$today, this.$yesterday, this.$older], "hideAll");

        return this;
    }
});
