AJS.test.require(["com.atlassian.confluence.plugins.recently-viewed-plugin:app-resources"], function () {
    var module = QUnit.module;
    var test = QUnit.test;

    module("RecentPageStorage", {
        setup: function() {

            this.addCurrentPage = function() {
                this.pageStorage.addCurrentPage({id: "123", title: "page 1", space: "my space"});
            };

            // simple mock local storage
            var storage = {
                items: {},
                getItem: function(key) {
                    return this.items[key] || null;
                },
                setItem: function(key, value) {
                    this.items[key] = value;
                }
            };

            this.pageStorage = new RY.RecentPageStorage(storage);
        }
    }, function () {
        test("add pages", function(assert) {
            assert.strictEqual(this.pageStorage.getPages().length, 0, "Should be empty");

            this.addCurrentPage();
            assert.strictEqual(this.pageStorage.getPages().length, 1, "Should have one page");

            // add it again
            this.addCurrentPage();
            assert.strictEqual(this.pageStorage.getPages().length, 1, "Should still have one page");
        });

        test("add page without title or id", function(assert) {
            this.pageStorage.addCurrentPage({});
            assert.strictEqual(this.pageStorage.getPages().length, 0, "Should not have been added");
        });
    });

    module("FilterView", {
        setup: function() {
            this.collection = new RY.PageCollection();
            var mockNavigationEvents = {
                trigger: function() {}
            };

            var $input = AJS.$("<div class='filter'><input class='filter-input' type='text'></div>");
            this.filterView = new RY.FilterView({el: $input, collection: this.collection, navigationEvents: mockNavigationEvents});
        }
    }, function () {
        test("search on input event", function (assert) {
            var done = assert.async();
            this.filterView.render();
            var spy = sinon.spy(this.collection, "search");
            this.filterView.$input.trigger($.Event("input"));

            setTimeout(function() {
                assert.ok(spy.calledOnce, "collection should have been searched");
                done();
            }, 100);
        });

        test("search on keyup event", function (assert) {
            var done = assert.async();
            this.filterView.render();
            var spy = sinon.spy(this.collection, "search");
            this.filterView.$input.trigger($.Event("keyup"));

            setTimeout(function() {
                assert.ok(spy.calledOnce, "collection should have been searched");
                done();
            }, 100);
        });

        test("search on input and keyup event", function (assert) {
            var done = assert.async();
            this.filterView.render();
            var spy = sinon.spy(this.collection, "search");

            // trigger both events!
            this.filterView.$input.trigger($.Event("input"));
            this.filterView.$input.trigger($.Event("keyup"));

            setTimeout(function() {
                assert.ok(spy.calledOnce, "collection should have only been searched once");
                done();
            }, 100);
        });
    });

    module("Page", {
        setup: function() {
            // make sure contextPath is not system dependent
            AJS.contextPath = function() {
                return "";
            };

            this.collection = new RY.PageCollection();
            this.collectionView = new RY.PageCollectionView({collection: this.collection});
            this.collectionView.render();

            this.page1 = new RY.Page({
                title: "page one",
                space: "space one",
                lastSeen: new Date()
            });

            this.page2 = new RY.Page({
                title: "page two",
                lastSeen: new Date()
            });

            this.page3 = new RY.Page({
                title: "page three",
                lastSeen: new Date()
            });

            this.collection.reset([this.page1, this.page2, this.page3]);
        }
    }, function () {
        test("PageCollection search", function(assert) {

            // search for 'one'
            var result = this.collection.search("one");
            assert.strictEqual(result.length, 1);
            assert.strictEqual(result[0], this.page1);

            // search for 'page'
            result = this.collection.search("page");
            assert.strictEqual(result.length, 3);

            // search for ''
            result = this.collection.search("");
            assert.strictEqual(result.length, 3);

            // search for combination of page name and space
            result = this.collection.search("space one page");
            assert.strictEqual(result.length, 1);
        });

        test("Search for space", function(assert) {
            var result = this.collection.search("space on");
            assert.strictEqual(result.length, 1);
            assert.strictEqual(result[0], this.page1);
        });
    });

    module("util", function () {
        test("daysSince", function(assert) {
            var now = new Date();
            var daysSince = RY.util.daysSince(now);
            assert.strictEqual(daysSince, 0);

            var yesterday = new Date(now.getFullYear(), now.getMonth(), now.getDate() - 1);
            yesterday.setMinutes(20);
            daysSince = RY.util.daysSince(yesterday);
            assert.strictEqual(daysSince, 1);

            var threeDaysAgo = new Date(now.getFullYear(), now.getMonth(), now.getDate() - 3);
            daysSince = RY.util.daysSince(threeDaysAgo);
            assert.strictEqual(daysSince, 3);
        });
    });
});
