define('confluence/page-hierarchy/service/data-service', [
    'ajs'
], function (AJS) {
    /**
     * This service is responsible for maintaining promise objects for various bits of data for a flow.
     * Once the flow is started, requests can be added to a clean data-service. They remain here for the lifecycle
     * of the flow, and any new requests can be added to it, but getting an existing request means that multiple
     * requests for the same piece of data do not trigger multiple Ajax requests, and promises can be added even
     * before they are requested by the user, so that bits of data can begin to be preloaded.
     */

    var requests = {};

    /**
     * Get an existing request, or add the request if it doesn't exist.
     * @param {string} key Key for the request
     * @param {jQuery.Deferred} promise Optional: Promise object.
     * @returns {jQuery.Deferred} The promise object
     */
    function get(key, promise) {
        if (!requests[key]) {
            if (!promise) {
                AJS.error('A request with key \'' + key + '\' does not exist, and one was not provided. ' +
                    'Returning a failed promise.');
                return AJS.$.Deferred().reject();
            }
            requests[key] = promise;
        }
        return requests[key];
    }

    /**
     * Clear the request cache
     * @returns {undefined}
     */
    function reset() {
        _abortAnyAjaxRequests();
        requests = {};
    }

    /**
     * Aborts any requests that have an abort method
     * @returns {undefined}
     * @private
     */
    function _abortAnyAjaxRequests() {
        for (var key in requests) {
            if (requests.hasOwnProperty(key) && typeof requests[key].abort === 'function') {
                requests[key].abort();
            }
        }
    }

    return {
        get: get,
        reset: reset,
        SUBTREE_COUNT_TOTAL_ONLY_DATA_KEY: 'subtree-count-total-only',
        SUBTREE_COUNT_DATA_KEY: 'subtree-count',
        INCOMING_LINKS_DATA_KEY: 'incoming-links',
        RESTRICTIONS_DATA_KEY: 'restrictions',
        SPACE_PERMISSIONS_DATA_KEY: 'space-permissions',
        ROOT_INCOMING_LINKS_COUNT_DATA_KEY: 'root-incoming-links'
    }
});