package com.atlassian.confluence.plugins.mentions.rest;

import com.atlassian.confluence.api.model.pagination.SimplePageRequest;
import com.atlassian.confluence.api.model.people.User;
import com.atlassian.confluence.api.service.network.NetworkService;
import com.atlassian.confluence.pages.AttachmentManager;
import com.atlassian.confluence.plugins.rest.entities.SearchResultEntity;
import com.atlassian.confluence.plugins.rest.entities.SearchResultEntityList;
import com.atlassian.confluence.plugins.rest.entities.builders.PersonalInformationContentEntityBuilder;
import com.atlassian.confluence.plugins.rest.manager.DateEntityFactory;
import com.atlassian.confluence.plugins.rest.manager.UserEntityHelper;
import com.atlassian.confluence.plugins.rest.resources.AbstractResource;
import com.atlassian.confluence.security.Permission;
import com.atlassian.confluence.security.PermissionManager;
import com.atlassian.confluence.security.SpacePermissionManager;
import com.atlassian.confluence.setup.settings.SettingsManager;
import com.atlassian.confluence.user.AuthenticatedUserThreadLocal;
import com.atlassian.confluence.user.ConfluenceUser;
import com.atlassian.confluence.user.PersonalInformation;
import com.atlassian.confluence.user.PersonalInformationManager;
import com.atlassian.confluence.user.UserAccessor;
import com.atlassian.confluence.user.avatar.AvatarProviderAccessor;
import com.atlassian.confluence.util.i18n.I18NBeanFactory;
import com.atlassian.plugin.webresource.WebResourceUrlProvider;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Iterables;

import javax.ws.rs.DefaultValue;
import javax.ws.rs.GET;
import javax.ws.rs.Path;
import javax.ws.rs.Produces;
import javax.ws.rs.QueryParam;
import javax.ws.rs.core.Response;

import java.util.Iterator;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

import static javax.ws.rs.core.MediaType.APPLICATION_JSON;

@Path("/network")
public class NetworkResource extends AbstractResource {
    private final PermissionManager permissionManager;
    private final PersonalInformationManager personalInformationManager;
    private final UserAccessor userAccessor;
    private final SettingsManager settingsManager;
    private final I18NBeanFactory i18NBeanFactory;
    private final WebResourceUrlProvider webResourceUrlProvider;
    private final DateEntityFactory dateEntityFactory;
    private final NetworkService networkService;
    private final AvatarProviderAccessor avatarProviderAccessor;
    private final AttachmentManager attachmentManager;

    public NetworkResource(
            final PermissionManager permissionManager,
            final UserAccessor userAccessor,
            final PersonalInformationManager personalInformationManager,
            final SettingsManager settingsManager,
            final I18NBeanFactory i18NBeanFactory,
            final SpacePermissionManager spacePermissionManager,
            final WebResourceUrlProvider webResourceUrlProvider,
            final DateEntityFactory dateEntityFactory,
            final NetworkService networkService,
            final AvatarProviderAccessor avatarProviderAccessor,
            final AttachmentManager attachmentManager) {
        super(userAccessor, spacePermissionManager);

        this.permissionManager = permissionManager;
        this.userAccessor = userAccessor;
        this.personalInformationManager = personalInformationManager;
        this.settingsManager = settingsManager;
        this.i18NBeanFactory = i18NBeanFactory;
        this.webResourceUrlProvider = webResourceUrlProvider;
        this.dateEntityFactory = dateEntityFactory;
        this.networkService = networkService;
        this.avatarProviderAccessor = avatarProviderAccessor;
        this.attachmentManager = attachmentManager;
    }

    @GET
    @Produces({APPLICATION_JSON})
    public Response doSearch(@QueryParam("max-results") @DefaultValue("10") Integer maxResults) {
        createRequestContext();

        // get the current user
        ConfluenceUser currentUser = AuthenticatedUserThreadLocal.get();

        // if the user is anonymous or doesn't have permission to view their network, return nothing
        if (currentUser == null ||
                !permissionManager.hasPermission(currentUser, Permission.VIEW, PermissionManager.TARGET_PEOPLE_DIRECTORY)) {
            return Response.noContent().build();
        }

        Set<User> items = new LinkedHashSet<>();
        // always show the current user first
        items.add(User.fromUsername(currentUser.getName()));
        // get users this user is following
        items.addAll(networkService.getFollowing(currentUser.getKey(), new SimplePageRequest(0, maxResults)).getResults());
        // if they aren't following many people, add some followers
        if (items.size() < maxResults) {
            final Iterator<User> followers = networkService.getFollowers(currentUser.getKey(), new SimplePageRequest(0, maxResults)).iterator();
            while (items.size() < maxResults && followers.hasNext()) {
                items.add(followers.next());
            }
        }
        // clamp results to size
        if (items.size() > maxResults) {
            items = ImmutableSet.copyOf(Iterables.limit(items, maxResults));
        }

        final PersonalInformationContentEntityBuilder entityBuilder = new PersonalInformationContentEntityBuilder(
                settingsManager,
                dateEntityFactory,
                userAccessor,
                new UserEntityHelper(userAccessor, settingsManager, webResourceUrlProvider, i18NBeanFactory),
                avatarProviderAccessor,
                attachmentManager
        );

        // construct the search results so we can build json out of it
        final List<SearchResultEntity> resultList = items.stream()
                .map(item -> userAccessor.getUserByName(item.getUsername()))
                .map(item -> {
                    PersonalInformation personalInformation = personalInformationManager.getOrCreatePersonalInformation(item);
                    return entityBuilder.build(personalInformation);
                })
                .collect(Collectors.toList());

        SearchResultEntityList searchResultEntityList = new SearchResultEntityList();
        searchResultEntityList.setResults(resultList);
        searchResultEntityList.setTotalSize(resultList.size());

        return Response.ok(searchResultEntityList).build();
    }

}
