package com.atlassian.confluence.plugins.mentions;

import com.atlassian.confluence.core.ContentEntityObject;
import com.atlassian.confluence.core.DataSourceFactory;
import com.atlassian.confluence.mail.notification.listeners.NotificationData;
import com.atlassian.confluence.pages.BlogPost;
import com.atlassian.confluence.pages.Comment;
import com.atlassian.confluence.pages.Page;
import com.atlassian.confluence.pages.TinyUrl;
import com.atlassian.confluence.plugin.descriptor.mail.NotificationContext;
import com.atlassian.confluence.user.ConfluenceUser;

import javax.activation.DataHandler;
import javax.activation.DataSource;

import static com.atlassian.confluence.renderer.radeox.macros.MacroUtils.defaultVelocityContext;
import static com.atlassian.confluence.util.GeneralUtil.getGlobalSettings;
import static org.apache.commons.lang3.StringUtils.isNotBlank;

public class NotificationDataServiceImpl implements NotificationDataService {
    private final DataSourceFactory imageDataSourceFactory;

    public NotificationDataServiceImpl(DataSourceFactory imageDataSourceFactory) {
        this.imageDataSourceFactory = imageDataSourceFactory;
    }

    @Override
    public NotificationData prepareDecorationContext(ConfluenceUser user, ContentEntityObject contentEntityObject) {
        NotificationData notificationData = new NotificationData(user, true, null);
        NotificationContext context = notificationData.getCommonContext();

        // Common data for webFragmentContext.
        context.setActor(user);
        context.setAction("mention");
        context.setContent(contentEntityObject);

        context.putAll(defaultVelocityContext());
        DataHandler avatarDataHandler = createAvatarDataHandler(user);
        context.put("avatarCid", avatarDataHandler.getName());
        context.addTemplateImage(createAvatarDataHandler(user).getDataSource());
        // Get the avatar, add its content ID to the template before it's rendered.
        context.put("sender", user);
        context.put("ceo", contentEntityObject);

        ContentEntityObject tinyUrlTarget = contentEntityObject;
        if (Comment.CONTENT_TYPE.equals(contentEntityObject.getType())) {
            Comment comment = (Comment) contentEntityObject;
            tinyUrlTarget = comment.getContainer();
            context.put("comment", contentEntityObject);
        } else if (Page.CONTENT_TYPE.equals(contentEntityObject.getType())) {
            context.put("page", contentEntityObject);
        } else if (BlogPost.CONTENT_TYPE.equals(contentEntityObject.getType())) {
            context.put("page", contentEntityObject);
        }
        String domainName = getGlobalSettings().getBaseUrl();
        if (isNotBlank(domainName) && domainName.endsWith("/")) {
            domainName = domainName.substring(0, domainName.length() - 1);
        }
        context.put("baseurl", domainName);
        if (tinyUrlTarget != null) {
            context.put("tinyUrl", domainName + "/x/" + new TinyUrl(tinyUrlTarget.getId()).getIdentifier());
        }

        return notificationData;
    }

    private DataHandler createAvatarDataHandler(ConfluenceUser user) {
        DataSource avatarDataSource = imageDataSourceFactory.getAvatar(user);
        return new DataHandler(avatarDataSource);
    }
}
