/**
 * Confluence Task List Plugin.
 *
 * Support for basic checkboxes is in tinymce core as list handling is tightly coupled into tiny.
 *
 * This provided additional support, such as supporting "checking", etc.
 *
 */
define('confluence-inline-tasks/tinymce/editor_plugin_src', [
    'jquery',
    'ajs',
    'tinymce',
    'confluence-editor/support/atlassian-editor-support'
], function(
    $,
    AJS,
    tinymce,
    atlassianEditorSupport
) {
    "use strict";

    var REGEX_SQUARE_OPEN_SQUARE_BRACKET = /^\u200b?(\[)$/;

    // aui/event
    AJS.bind("confluence.editor.registerHandlers", function(e, data) {
        if (!atlassianEditorSupport.inlineTasks()) {
            return;
        }

        var handlerManager = data.handlerManager;
        var ed = data.ed;
        var createHandler = data.createHandler;

        handlerManager.registerHandler(
            "]".charCodeAt(0),
            createHandler(REGEX_SQUARE_OPEN_SQUARE_BRACKET, function () {
                ed.plugins.lists.applyList('UL', 'OL', 'inline-task-list', {
                    attributesOnItems: {'data-inline-task-id' : ''},
                    // aui/i18n
                    placeholderText : AJS.I18n.getText("inline-tasks.editor.placeholder.inline.tasks")
                });
            },
            true
        ));
    });

    /**
     * Override "ed.plugins.lists.applyList"
     * Instead of touch "ed.plugins.lists.applyList" in editor core in "/atlassian-tinymce/src/main/resources/jscripts/tiny_mce/plugins/lists/editor_plugin_src.js"
     * It is quite dangerous to change too much in core
     * Moreover, the CONFDEV-22940 happens in inline-task only.
     * Therefore, it is better to change in inline-task plugin
     * @param ed tinymce editor object
     */
    function overrideApplyList(ed){
        ed.plugins.lists.applyList = _.bind(_.wrap(ed.plugins.lists.applyList, function(oldApplyList) {
            oldApplyList.apply(this, Array.prototype.slice.call(arguments, 1));
            makeSureLiNotEmptyAtCursor(ed);
        }), ed.plugins.lists);
    }

    /**
     * CONFDEV-22940
     * In FF, Safari and IE8+11, empty <li> will have 2 issues, one of reasons is li[data-inline-task-id] has "list-style": none
     * - 1. Empty list does not appear, it is collapsed
     * - 2. Can not navigate cursor by Up/Down keys or use mouse click on empty list
     *
     * Therefore this function makes new <li> should have at least a space character
     * @param ed
     * @param $li optional jQuery collection of <li>
     */
    function makeSureLiNotEmptyAtCursor(ed, $li){

        if (!(tinymce.isIE8 || tinymce.isIE11 || tinymce.isGecko || tinymce.isWebKit)) {
            return;
        }

        var $list = $li || $(getCurrentListItem(ed));

        //in IE8, though empty <li> always have height>0, background image still does not appear
        //so we have to use other approach to process
        if ($list.height() === 0 ||
            tinymce.isIE8) {

            $list.append("&nbsp;");
            (tinymce.isWebKit || tinymce.isIE8) && ed.selection.setCursorLocation($list[0], 0);
        }
    }

    /* This is very dependant on the images used and the padding, etc */
    function isInCheckArea(e, el) {
        var offsetX;
        var offsetY;
        var clientRect;

        // Browsers are all over the place for mouse events. :/
        if(tinymce.isGecko) {
            clientRect = el.getBoundingClientRect();
            // We add the current scrolling position to clientRect because its values are relative to the viewport
            offsetX = e.pageX - (clientRect.left + tinymce.DOM.getViewPort(tinymce.activeEditor.getWin()).x);
            offsetY = e.pageY - (clientRect.top + tinymce.DOM.getViewPort(tinymce.activeEditor.getWin()).y);
        } else if(tinymce.isIE8) {
            offsetX = e.offsetX - el.offsetLeft;
            offsetY = e.offsetY - el.offsetTop;
        } else {
            offsetX = e.offsetX;
            offsetY = e.offsetY;
        }

        return offsetX >= 3 && offsetX <= 14
            && offsetY >= 3 && offsetY <= 14;
    }

    function handleTaskListCheck(ed, e) {
        var el = e.target;

        if (el.tagName !== 'LI') {
            return;
        }
        var list = ed.dom.getParent(el, "ul,ol");

        if(list.tagName === 'UL' && ed.dom.hasClass(list, 'inline-task-list')) {
            if(isInCheckArea(e, el)) {
                if(ed.dom.hasClass(el, 'checked')) {
                    ed.dom.removeClass(el, 'checked');
                } else {
                    ed.dom.addClass(el, 'checked');
                }
            }
        }
    }

    function getCurrentListItem(ed) {
        var el = ed.selection.getStart();

        // Get start will return BR if the LI only contains a BR or an empty element as we use these to fix caret position
        var li = (el && el.tagName && el.tagName.toUpperCase() === 'LI')
                    ? el
                    : ed.dom.getParent(el, 'li');

        return li;
    }

    function handleNewListItem(ed, e) {
        if (e.keyCode === tinymce.VK.ENTER && !e.shiftKey) {
            var li = getCurrentListItem(ed);
            if (li) {
                var $li = $(li);
                if (ed.dom.hasClass(li.parentNode, 'inline-task-list')) {
                    // Set the id to empty, so that it can be recognized when pasting <li> elements.
                    $li.removeClass('checked').attr('data-inline-task-id', '');
                } else {
                    $li.removeClass('checked').removeAttr('data-inline-task-id');
                }
                makeSureLiNotEmptyAtCursor(ed, $li);
            }
        }
    }

    // CONFDEV-23427 Font tags get inserted into <li>s in IE, so items inserted after completed tasks
    // appear grayed out. Remove these font tags as a fix.
    function remoteExtraFontTagsForTasks(ed, e) {
        var li = getCurrentListItem(ed);
        if (li) {
            var $li = $(li);
            if (ed.dom.hasClass(li.parentNode, 'inline-task-list')) {
                $li.find("font").contents().unwrap();
            }
        }
    }

    return {
        init : function(ed, url) {
            overrideApplyList(ed);
            ed.onClick.add(handleTaskListCheck);
            ed.onKeyUp.add(handleNewListItem); // Needs to be onKeyUp so that newly list item has been added
            if (tinymce.isIE) {
                ed.onKeyUp.add(remoteExtraFontTagsForTasks);
            }
        },

        getInfo : function() {
            return {
                longname : 'Confluence Task List Plugin',
                author : 'Atlassian',
                authorurl : 'http://www.atlassian.com',
                version : tinymce.majorVersion + "." + tinymce.minorVersion
            };
        }
    };
});

require('confluence/module-exporter').safeRequire('confluence-inline-tasks/tinymce/editor_plugin_src', function(InlineTasks) {
    var tinymce = require('tinymce');
    var tinymceBootstrap = require('confluence-editor/loader/tinymce-bootstrap');

    tinymce.create('tinymce.plugins.ConfluenceTaskListPlugin', InlineTasks);
    tinymce.PluginManager.add('confluencetasklist', tinymce.plugins.ConfluenceTaskListPlugin);

    tinymceBootstrap.addTinyMcePluginInit(function(settings) {
        settings.plugins += ",confluencetasklist";
    });
});
