(function ($) {
    "use strict";

    /**
     * Constructor function of TasksTableSortable.
     * All possible options are listed in this constructor function
     * @param options
     * @constructor
     */
    var TasksTableSortable = function (options) {
        this.ajaxUrl = options.ajaxUrl;
        this.restUrlPagination = options.restUrlPagination;

        //$wrapper is a element that includes table and pagination elements
        this.$wrapper = options.$wrapper;
        this.table = options.table;
        this.$table = $(this.table);

        this.analyticEventKey = options.analyticEventKey;

        //som options for sorting to work
        this.sortColumnDefault = options.sortColumnDefault || "duedate";
        this.sortReverseSortDefault = options.sortReverseSortDefault || false;
        this.reportParametersDefault = options.reportParametersDefault;

        //some options for pagination
        this.pageIndex = options.pageIndex || 0;
        this.pageSize = options.pageSize || 10;
        this.pageTotal = options.pageTotal || 0;
        this.pageLimit = options.pageLimit || 7;
        this.adaptive = options.adaptive;
        this.columns = options.columns;

        this.templates = options.templates;

        //some callbacks
        this.onRenderEmptyTable = options.onRenderEmptyTable;
        this.onBusySorting = options.onBusySorting;
    };

     /**
     * Map value of sortColumn which is used by server-side to sort
     * to column name (is value of data-column-name of "table > thead > th")
     * @param sortBy
     * @returns {*}
     */
    TasksTableSortable.getColumnNameFromSortBy = function (sortBy) {
        var columnMap = {
            "due date": "duedate",
            "page title": "location",
            "assignee": "assignee"
        };
        return columnMap[sortBy] ? columnMap[sortBy] : "duedate";
    };

    /**
     * Map column name (is value of data-column-name of "table > thead > th")
     * to value of sortColumn which is used by server-side to sort
     * @param columnName
     * @returns {*}
     */
    TasksTableSortable.getSortByFromColumnName = function (columnName) {
        var columnMap = {
            duedate: "due date",
            location: "page title"
        };
        return columnMap[columnName] ? columnMap[columnName] : columnName;
    };

    TasksTableSortable.prototype.updateOptions = function (options) {
        $.extend(this, options);
        this.$table = $(this.table);
    };

     /**
     * Get current page index. So far we get page index from DOM tree that
     * Confluence.UI.Components.Pagination render already
     * @param table
     * @returns {Number}
     */
    TasksTableSortable.prototype.getCurrentPageIndex = function () {
        //TODO: there is a bug in Confluence.UI.Components.Pagination
        //CONFDEV-24789: it renders ol.macro-auto-pagination 2 times and nested together
        var $pagination = this.$wrapper.find(".macro-auto-pagination").last();
        var pageIndex = parseInt($pagination.attr("data-initial-page-index"), 10);
        return isNaN(pageIndex) ? 0 : pageIndex;
    };

    /**
     * Render pagination basing on custom data
     */
    TasksTableSortable.prototype.renderPagination = function ($scope, updatedReportParams) {
        var self = this;

        if (!$scope) {
            $scope = self.$table;
        }

        if (!updatedReportParams) {
            updatedReportParams = self.reportParametersDefault;
        }

        this.$wrapper.find(".macro-auto-pagination").remove();

        if (!(self.pageTotal > 1)) {
            return;
        }

        Confluence.UI.Components.Pagination.build({
            scope: $scope,
            pageSize: self.pageSize,
            totalPages: self.pageTotal,
            pageLimit: self.pageLimit,
            path: self.restUrlPagination,
            adaptive: self.adaptive,
            currentPage: self.pageIndex,
            data: {
                reportParameters: JSON.stringify(updatedReportParams)
            },

            success: function addLine(line, $container) {
                var lineData = {
                    'task': line,
                    'columns': self.columns
                };
                var renderedLine = self.templates.tasksReportLine(lineData);
                $container.append(renderedLine);
            }
        });
    };


    TasksTableSortable.prototype.toggleBusyState = function (isBusy) {
        // for testability
        this.$wrapper.attr("data-loading", isBusy);

        if (isBusy) {
            this.$wrapper.find('.task-blanket').show();
        }
        else {
            this.$wrapper.find('.task-blanket').hide();
        }

        if (typeof this.onBusySorting === "function") {
            this.onBusySorting.apply(this, [isBusy]);
        }
    };

    TasksTableSortable.prototype.renderTable = function (rowData) {
          //build html of rows
        var self = this;
        var htmlRows = _.map(rowData, function(taskData) {
            return self.templates.tasksReportLine({
                'task': taskData,
                'columns': self.columns
            });
        }).join("");
        self.$table.find("tbody").html(htmlRows);

        Confluence.Binder.userHover();
    };

    TasksTableSortable.prototype._triggerAnalyticsSorting = function () {
        var eventKey = this.analyticEventKey;
        var data = {
            column: this.sortColumn,
            direction: this.reverseSort ? "desc" : "asc"
        };

        AJS.trigger('analyticsEvent', {name: eventKey, data : data});
    };

    TasksTableSortable.prototype._buildAjaxData = function (reportParams) {
        var ajaxObject = {
            url: this.ajaxUrl,
            cache: false,
            dataType: 'json',
            data: {
                pageIndex: this.pageIndex,
                pageSize: this.pageSize,
                reportParameters: JSON.stringify(reportParams)
            }
        };
        return ajaxObject;
    };

    TasksTableSortable.prototype.init = function () {
        var self = this;

        self.sortColumn = self.sortColumnDefault;
        self.reverseSort = self.sortReverseSortDefault;

        this.$table.tablesorterServerOnly({
            sortColumn: self.sortColumn,
            reverseSort: self.reverseSort,

            onInit: function () {
                var $table = $(this);

                $table.addClass("aui-table-sortable"); //re-use style from AUI
            },

            onSort: function (p) {
                var table = this;
                var $table = $(table);

                self.pageIndex = self.getCurrentPageIndex();
                self.sortColumn = p.sortColumn;
                self.reverseSort = p.reverseSort;

                self.toggleBusyState(true);

                var updatedReportParams = $.extend({}, self.reportParametersDefault, {
                    sortColumn: TasksTableSortable.getSortByFromColumnName(self.sortColumn),
                    reverseSort: self.reverseSort
                });

                var ajaxObject = self._buildAjaxData(updatedReportParams);

                $.ajax(ajaxObject)
                    .done(function (data) {
                        self.pageIndex = self.getCurrentPageIndex();
                        self.pageTotal = data.totalPages;

                        //When there is no data, should hide pagination
                        if (self.pageIndex === 0 &&
                                self.pageTotal === 0) {

                            if (typeof self.onRenderEmptyTable === "function") {
                                self.$wrapper.find(".macro-auto-pagination").remove();
                                $table.remove();
                                self.onRenderEmptyTable.apply(self);
                            }

                            return;
                        }

                        self.renderTable(data.detailLines);

                        self.renderPagination(null, updatedReportParams);
                        $table.trigger("refreshHeader.sortServerOnly");
                        self._triggerAnalyticsSorting();
                    })
                    .fail(function() {
                        var notice = new Confluence.InlineTasks.Notice({
                            textMessage: AJS.I18n.getText("inline-tasks.macro.tasks-report.sort.error"),
                            className: "forbidden-notice"
                        });
                        notice.show();
                    })
                    .always(function () {
                        // HACK: CONFSERVER-53278 Rendering the content after sorting table content with JIRA issue macros
                        AJS.trigger("ic-jim-async-supported");
                        self.toggleBusyState(false);
                    });
            }
        });
    };

    //expose to outside
    var Confluence = window.Confluence || {};
    Confluence.InlineTasks = Confluence.InlineTasks || {};
    Confluence.InlineTasks.TasksTableSortable = TasksTableSortable;
})(AJS.$);

