define('confluence/ic/view/sidebar',
    [
        'underscore',
        'backbone',
        'jquery',
        'confluence/ic/view/display-comment',
        'confluence/ic/view/create-comment',
        'confluence/ic/util/utils',
        'ajs'
    ],
    function(
        _,
        Backbone,
        $,
        DisplayCommentView,
        CreateCommentView,
        Utils,
        AJS
    ) {

    "use strict";

    var SidebarView = Backbone.View.extend({
        tagName: "div",

        className: "ic-sidebar",

        events: {
            "click button.ic-action-hide" : function(e) {
                this.hide(e, true);
                Backbone.trigger("ic:sidebar:close");
            }
        },

        initialize: function(options) {
            var me = this;
            this.listenTo(Backbone, "ic:view ic:overlap", this.displayComment);
            this.listenTo(Backbone, "ic:create", this.displayCreateComment);
            this.listenTo(Backbone, "ic:error", this.displayError);
            this.listenTo(Backbone, "ic:hide-sidebar", this.hide);
            this.listenTo(Backbone, "ic:clearError", this.clearError);

            if (Utils.getDarkFeatures().RICH_TEXT_EDITOR) {

                AJS.bind('rte-quick-edit-ready', function() {
                    // Editor element contain iframe which make problem with MutationObserver,
                    // this code force MutationObserver re-update to correct size.
                    var editorIframe = me.$el.find('#wysiwygTextarea_ifr');
                    if (editorIframe.length !== 0) { // RTE inside our sidebar
                        var editorObserver = new MutationObserver(function() {
                            editorIframe.resize();
                        });
                        editorObserver.observe(editorIframe[0], {attributes: true, attributeFilter: ["style"]});
                        if (!AJS.Rte.getEditor().getContent()) {
                            me._scrollToEditorView(editorIframe);
                        }
                    } else { // RTE in page comment or page edit
                        // In-case user active quick-edit for page when viewing an inline-comment, then $content's padding should be re-set
                        var isShowing = me.$el.is(':visible');
                        if (!isShowing) {
                            Utils.getPageContainer().css("padding-right", "0");
                            me.$wikiContent.css("position", "static");

                            // min-height is set when we push down the footer so it won't overflow our sidebar,
                            // reset min-height to default value if not Edit Page won't has a full height
                            Utils.getPageContainer().css({ "min-height": 0 });
                        }
                    }
                });
            } else {
                // hide the sidebar when editing a page
                AJS.bind('init.rte', _.bind(this.hideInEditMode, this));
            }

            this.defaultNavOffset = 130;
            this.defaultPermalinkedCommentOffset = 100;
            this.$wikiContent = options.$wikiContent || $("#content .wiki-content").first();

            // This is needed because many interaction resize the sidebar Edit/Delete/Save/Add Reply/ etc
            // Impossible to manually resize the sidebar in all parts of code. This will do it for us.
            this.sidebarObserver = new MutationObserver(_.bind(this.setSidebarYPos, this));
            this.sidebarObserver.observe(this.el, { childList: true, subtree: true });

            // Reposition the inline comment when resizing the content
            $(window).resize(_.debounce(_.bind(this.setSidebarYPos, this), 100));
            AJS.bind('sidebar.expanded sidebar.collapsed', _.debounce(_.bind(this.setSidebarYPos, this), 50));
        },

        /**
         * @param comment the comment model to be displayed
         * @param source the source that triggers displaying comment.
         *        'nav' for navigation in sidebar,
         *        'permalink' for permalinked comment,
         *        'quickreload' for quick reload show comment
         *        undefined if triggering from anywhere else, like clicking a highlight on the page
         * @param options
         *          options.isReplyComment if TRUE, reply editor will be activated automatically
         *          options.ignoreConfirmDialog specify FALSE value to ignore showing confirm message
         */
        displayComment: function(comment, source, options) {
            var doDisplay = function() {
                if (comment.highlight.is(':hidden')) {
                    Utils.showHighlightContent(comment.highlight);
                    if (comment.highlight.is(':hidden')) {
                        return;
                    }
                }

                this.distanceToTop = this.previousHighlightOffset || this.defaultNavOffset;
                this._emptySidebar(true);
                this.commentView = new DisplayCommentView({
                    model: comment,
                    source : source
                });

                this.show(true);

                this.previousHighlightOffset = comment.highlight.position().top;
                var $commentViewEl = this.commentView.render().$el;
                //hidden view in the first time and show it when we completed calculating the position of view
                $commentViewEl.css('visibility', 'hidden');

                if (source === 'nav') {
                    this._scrollToComment(comment);
                }
                else if (source === 'permalink') {
                    this._scrollToPermalinkedComment(comment, this.defaultPermalinkedCommentOffset, options);
                }
                else if (source === 'quickreload') {
                    this._scrollToQuickReloadComment(comment);
                }
                this.$el.html($commentViewEl);
                Confluence.Binder && Confluence.Binder.userHover();
            };
            var needToShowConfirmDialog = (options === undefined) || (options.ignoreConfirmDialog === false);

            if (needToShowConfirmDialog) {
                if (Utils.confirmProcess(true)) {
                    doDisplay.call(this);
                }
            } else {
                doDisplay.call(this);
            }
            // HACK!!! for render JIM asynchronous, first times loading side panel and comments
            AJS.trigger("ic-jim-async-supported");
        },

        _scrollToQuickReloadComment: function(comment) {
            var $feedbackNotice = $('.qr-feedback:first');
            var spaceFromTopOfWindow = this.defaultPermalinkedCommentOffset;
            if ($feedbackNotice.length > 0) {
                spaceFromTopOfWindow = $feedbackNotice.height() + 67;
            }
            this._scrollToPermalinkedComment(comment, spaceFromTopOfWindow);
        },

        _scrollToEditorView: function($editorElement) {
            var $bodyContentEl = Utils.isDocTheme() ? Utils.getSplitterContent() : $('body, html');
            var sidebarTopPosition = this.$('.ic-display-comment-view').position().top;
            var commentviewTop = $editorElement.closest('.ic-create-comment-view').position().top + sidebarTopPosition;
            if ($bodyContentEl.height() < commentviewTop || $bodyContentEl.scrollTop() > commentviewTop) {
                $bodyContentEl.animate({
                    scrollTop: commentviewTop
                }, 100);
            }
        },

        // Todo: move scrolling logic out of sidebar.js and into a Util library
        // Who can make the scrolling logic easier??? HELP!
        _scrollToPermalinkedComment: function(comment, spaceFromTopOfWindow, options) {
            var scrollOffset;
            if (Utils.isDocTheme()) {
                var applicationHeaderHeight = 40;
                scrollOffset = comment.highlight.offset().top + Utils.getSplitterContent().scrollTop() - applicationHeaderHeight;
                Utils.getSplitterContent().animate({
                    scrollTop: scrollOffset - spaceFromTopOfWindow
                }, 100).promise().done(_.bind(this._scrollToPermalinkedCommentCompleted, this, options));
            } else {
                scrollOffset = comment.highlight.offset().top;
                $('html, body').animate({
                    scrollTop: scrollOffset - spaceFromTopOfWindow
                }, 100).promise().done(_.bind(this._scrollToPermalinkedCommentCompleted, this, options));
            }
        },

        _scrollToPermalinkedCommentCompleted: function(options) {
            // if this is reply comment then we should activated the editor
            if (options && options.isReplyComment) {
                this.commentView.replyComment();
            }
        },

        // Todo: Move scrolling logic out of sidebar.js and into a Util library. Try to remove DOM dependencies
        // passing only offsets/positions instead, then write some Karma unit tests to assert this logic behaves
        // as expected
        _scrollToComment : function(comment) {
            var highlightTopOffset = comment.highlight.position().top;
            if (Utils.isDocTheme()) {
                Utils.getSplitterContent().animate({
                    scrollTop: Utils.getSplitterContent().scrollTop() + highlightTopOffset - this.distanceToTop
                }, 100);
            } else {
                $('html, body').animate({
                    scrollTop: window.pageYOffset + highlightTopOffset - this.distanceToTop
                }, 100);
            }
        },

        displayCreateComment: function(commentCollection, selectionObject, serializedHighlights) {
            var isOnUnsupportedMacro = this.isOnUnsupportedMacro();
            this._emptySidebar(true);
            this.createCommentView = new CreateCommentView({
                collection: commentCollection,
                selection: selectionObject,
                type: isOnUnsupportedMacro ? "pageComment" : "topLevel",
                isFocus: false,
                stopFocus:true,
                commentText: isOnUnsupportedMacro ? '<blockquote>' + AJS.escapeHtml(selectionObject.text) + '</blockquote> <div/>': "",
                // This function will be called whenever RTE in createCommentView was destroyed, there are three cases:
                // Case 1. createCommentView is showing, continues to highlight another text to create new IC
                // Case 2. createCommentView is showing, click to create/reply/edit a page level comment.
                // Case 3. click hide sidebar button
                onFinished: _.bind(function() {
                    if (!this.createCommentView.keepSidebar) { // In case 2, we need to hide the sidebar and clean the highlight
                        if (this.isShowing) {
                            this.hide();
                        }
                    }
                }, this),
                serializedHighlights: serializedHighlights
            });
            this.show();
            this.createCommentView.$el.css('visibility', 'hidden');
            this.$el.html(this.createCommentView.$el);
            this.createCommentView.render();
        },

        isOnUnsupportedMacro: function() {
            var $highlightText = $('.ic-current-selection');
            return Utils.containUnsupportedMacro($highlightText)
                    || Utils.containInternalLink($highlightText)
                    || Utils.containUserMetion($highlightText);
        },

        displayError: function(error, options) {
            options = _.extend({ closeable: true }, options);
            this.clearError();
            AJS.messages.warning(".ic-error", {
                body: error,
                closeable: options.closeable
            });
            options.callback && options.callback();
        },

        clearError: function() {
            $('.ic-error').empty();
        },

        show: function(isAnimation) {
            if (this.isShowing) {
                return;
            }
            this._toggleSidebar(isAnimation);
            this.isShowing = true;
        },

        hideInEditMode: function(e) {
            if (AJS.Meta.get('content-type') !== 'comment') {
                this.sidebarObserver.disconnect();
                this.hide(e);
            }
        },

        hide: function(e, isAnimation) {
            e && e.preventDefault();

            if (Utils.confirmProcess(true)) {
                if (!this.isShowing) {
                    return;
                }

                // when hiding the sidebar, set the page height back to initial because we may have overflowed the
                // page with a comment thread
                Utils.getPageContainer().css({ "min-height": "initial" });
                this._toggleSidebar(isAnimation);

                // toggleSidebar() needs to be executed first because emptySidebar will remove the active highlight
                // which is needed to maintain the scroll position of the page
                Backbone.trigger("ic:sidebar-hidden");
                this.isShowing = false;
            }
        },

        /*
         * maintains the scroll position in relation to text highlight while opening and closing the sidebar
         */
        _toggleSidebar : function(isAnimation) {
            var thiz = this;
            var activeHighlight = $('.inline-comment-marker.active:first, .ic-current-selection:first');
            var heightToTopOffset;
            var scrollOffset = (Utils.isDocTheme() ? Utils.getSplitterContent().scrollTop() : window.pageYOffset);
            if (activeHighlight.length > 0) {
                heightToTopOffset = activeHighlight.offset().top - scrollOffset;
            }

            var deferred = $.Deferred();
            var promise = deferred.promise();
            if (this.isShowing) { // collapse the sidebar
                if(isAnimation && Utils.isAnimationSupported()) {
                    promise = Utils.animateSidebar(this, false);
                } else {
                    Utils.getPageContainer().css("padding-right", "0");
                    this.$wikiContent.css("position", "static");
                    this._emptySidebar();
                    deferred.resolve();
                }
            } else {
                Utils.getPageContainer().css("padding-right", "280px");
                if(isAnimation && Utils.isAnimationSupported()) {
                    promise = Utils.animateSidebar(this, true);
                } else {
                    deferred.resolve();
                }
                this.$wikiContent.css("position", "relative");
            }

            promise.done(function() {
                Utils.scrollToCommentAfterToggleSideBar(heightToTopOffset, activeHighlight);
            });
        },

        _emptySidebar: function(keepSidebar) {
            if (this.commentView) {
                this.commentView.model.unset && this.commentView.model.unset("active");
                this.commentView.destroy(); // call destroy instead of remove() to clean up menu event handlers
                this.commentView = null;
            }

            if (this.createCommentView) {
                // keepSidebar value will let onFinished callback understand if sidebar need to be hidden or not.
                this.createCommentView.keepSidebar = keepSidebar;
                this.createCommentView.destroy();
                this.createCommentView = null;
            }
        },

        /*
         * This method is called whenever the content of the sidebar changes. It serves two purposes: one is to
         * vertically align the currently displayed comment or current create comment form with the corresponding
         * highlight. The second function is to expand the page height dynamically to keep the sidebar from overflowing
         * the page.
         */
        setSidebarYPos: function() {
            if (!this.isShowing) {
                return;
            }
            // when ic-sidebar is first initialized, its top value is not set so it gets bound to the upper relative parent,
            // we need to subtract this relative offset later on
            var initialOffset = this.$el.offset().top;
            var $viewEl, $highlightEl;
            if (this.commentView != null) {
                $viewEl = this.commentView.$el;
                $highlightEl = this.commentView.model.highlight;
            } else if (this.createCommentView != null) {
                $viewEl = this.createCommentView.$el;
                $highlightEl = $('.ic-current-selection');
            } else {
                return;
            }

            if (!$highlightEl) return;

            var sidebarHeaderHeight = this.$('.ic-sidebar-heading').height();
            $viewEl.css({ 'visibility': 'visible', position: "absolute", top: ($highlightEl.offset().top - initialOffset - sidebarHeaderHeight) + "px"});

            Utils.recalculateContentHeight($viewEl);
        }
    });

    return SidebarView;
});