define('confluence/ic/view/display-comment',
    [
        'jquery',
        'ajs',
        'underscore',
        'backbone',
        'confluence/ic/util/utils',
        'confluence/ic/view/create-comment',
        'confluence/ic/view/confirm-dialog',
        'confluence/ic/view/reply-list',
        'confluence/ic/view/resolved-dialog-discovery',
        'confluence/ic/view/navigation',
        'confluence/ic/view/likes'
    ],
    function(
        $,
        AJS,
        _,
        Backbone,
        Utils,
        CreateCommentView,
        ConfirmDialog,
        ReplyListView,
        ResolvedDialogDiscoveryView,
        NavigationView,
        LikesView
    ) {

    "use strict";

    var DisplayCommentView = Backbone.View.extend({
        tagName: "div",

        className: "ic-display-comment-view",

        template: Confluence.Templates.IC.displayComment,

        events: {
            "click .ic-action-reply" : "replyComment",
            "click .ic-action-resolve" : "resolveComment",
            "click .ic-action-edit" : "editComment",
            "click #show-resolved-comments" : "showResolvedDialogDiscovery",
            "click .ic-reply-placeholder .textarea" : "replyComment",
            "focus .ic-reply-container .textarea" : "onReplyPlaceholderFocus"
        },

        initialize: function(options) {
            // need to bind event outside of view because aui-dropdown2 is appended to body element
            _.bindAll(this, "confirm", "destroy", "deleteFail", "replyComment", "resolveComment", "editComment",
                "renderCommentOnly");

            this.model.set("active", true);
            this.listenTo(this.model, "change:resolveProperties", this.render);

            $('body').on('click', '.ic-action-delete a', this.confirm);
            $('body').on('click', '.ic-action-reply a', this.replyComment);
            $('body').on('click', '.ic-action-resolve a', this.resolveComment);
            $('body').on('click', '.ic-action-edit a', this.editComment);

            this.replyListView = new ReplyListView({
                commentModel: this.model,
                source: options.source
            });
        },

        render: function() {
            this.$el.html(this.template(
                _.extend(this.model.toJSON(), {
                    showMenu: true,
                    currentAuthorUserName: AJS.Meta.get('remote-user'),
                    currentAuthorAvatarUrl: Utils.getAuthorAvatarUrl(),
                    currentAuthorDisplayName: Utils.getAuthorDisplayName(),
                    darkFeatures: Utils.getDarkFeatures()
                })
            ));

            // Don't need these if comment was resolved.
            if (this.model.isResolved()) {
                // HACKKKK!!!
                this.$('.ic-comment-container').addClass('ic-resolved');
            } else {
                this.$('.ic-list').append(this.replyListView.render().el);
            }

            this._renderNavigation();
            this._renderLikes();
            Utils.addSidebarHeadingTooltip(this);

            return this;
        },

        _renderNavigation: function() {
            if (this.navigationView) {
                this.navigationView.remove();
            }
            this.navigationView = new NavigationView({ model: this.model });
            this.navigationView.render();
            this.$('#ic-nav-container').html(this.navigationView.el);
        },

        _renderLikes: function() {
            if (this.likesView) {
                this.likesView.remove();
            }
            this.likesView = new LikesView({contentId: this.model.id}).render();
            this.$('.ic-action-like').html(this.likesView.el);
        },

        editComment: function(e) {
            e.preventDefault();

            var that = this;
            if (this.editView) {
                this.editView.focusEditor();
            } else {
                if (Utils.confirmProcess()) {
                    var promise = Utils.getEditorFormat(this.model.get("id"));
                    promise.done(function(editBody) {
                        that.editView = new CreateCommentView({
                            model: that.model,
                            type: "edit",
                            selection: {},
                            onFinished: that.renderCommentOnly,
                            commentText: editBody
                        });
                        that.$('.ic-display-comment').html(that.editView.$el);
                        that.editView.render();
                    }).fail(function() {
                        Backbone.trigger("ic:error", AJS.I18n.getText('inline.comments.server.error.edit.comment'));
                    });
                }
            }
        },

        renderCommentOnly: function() {
            // destroy editView
            this.editView && this.editView.destroy();
            this.editView = null;
            
            var commentData = _.extend(this.model.toJSON(), {darkFeatures: Utils.getDarkFeatures()});
            this.$('.ic-display-comment').html(
                $(Confluence.Templates.IC.comment(commentData)).children());
            this._renderLikes();
        },

        replyComment: function(e) {
            e && e.preventDefault();

            if (this.createReplyView) {
                this.createReplyView.focusEditor();
            } else {
                if (Utils.confirmProcess()) {
                    this.createReplyView = new CreateCommentView({
                        model: this.model,
                        type: 'reply',
                        onFinished: _.bind(function() {
                            this.createReplyView && this.createReplyView.destroy();
                            this.createReplyView = null;
                        }, this)
                    });
                    this.$(".ic-reply-container").prepend(this.createReplyView.$el);
                    this.createReplyView.render(); // render after appending so we can focus the reply textarea
                } else {
                    // call blur to avoid placeholder has focus after click on cancel button from confirm dialog
                    e && e.currentTarget.blur();
                }
            }
        },

        confirm: function(e) {
            e.preventDefault();
            if (this.model.replies.length > 0) {
                Backbone.trigger("ic:error", AJS.I18n.getText('inline.comments.delete.with.reply'));
                return false;
            }
            var confirmDeleteDialog = new ConfirmDialog({
                model: new Backbone.Model({
                    label: AJS.I18n.getText('inline.comments.sidebar.delete'),
                    message: AJS.I18n.getText('inline.comments.delete.confirmation.message')
                }),
                $menuEl: $('.ic-action-menu'),
                $bodyEl: $('body')
            });

            this.$('.ic-sidebar-heading').append(confirmDeleteDialog.$el);
            this.listenTo(confirmDeleteDialog, "ic:confirm", this.deleteComment);
        },

        deleteComment: function() {
            var success = _.bind(function() {
                this.$el.html(Confluence.Templates.IC.deleteComment());
                this.model.set('deleted', true);
                this._renderNavigation();
                Utils.removeInlineLinksDialog(this.model.highlights);
                Utils.addSidebarHeadingTooltip(this);
                Backbone.trigger("ic:delete");
            }, this);

            this.model.destroy({
                wait: true,
                success: success,
                error: this.deleteFail
            });
        },

        deleteFail: function(model, response, options) {
            if (response.status === 401) {
                Backbone.trigger("ic:error", AJS.I18n.getText('inline.comments.delete.permission'));
            } else if (response.status === 409) { //DELETE_FAILED
                Backbone.trigger("ic:error", response.responseText);
            } else {
                Backbone.trigger("ic:error", AJS.I18n.getText('inline.comments.server.error'));
            }
        },

        destroy: function(isDeleteComment) {
            // need to clean up menu event handlers outside of the view
            $('body').off('click', '.ic-action-delete a', this.confirm);
            $('body').off('click', '.ic-action-reply a', this.replyComment);
            $('body').off('click', '.ic-action-resolve a', this.resolveComment);
            $('body').off('click', '.ic-action-edit a', this.editComment);
            this.createReplyView && this.createReplyView.destroy();
            this.editView && this.editView.destroy();
            this.replyListView.destroy();
            this.navigationView && this.navigationView.remove();
            this.likesView && this.likesView.remove();
            isDeleteComment !== true && this.remove();
        },

        resolveComment: function(e) {
            e && e.preventDefault();
            if (Utils.confirmProcess(true)) {
                this.model.resolve(true, {
                    wait: true,
                    success: _.bind(this.onResolveComplete, this),
                    error: this.onResolveFail
                });
            }
        },

        onResolveComplete: function() {
            this.createReplyView && this.createReplyView.destroy();
            this.createReplyView = null;

            Backbone.trigger("ic:resolved");
            Utils.removeInlineLinksDialog(this.model.highlights);
        },

        onResolveFail: function(model, response) {
            if (response.status === 401) {
                Backbone.trigger("ic:error", AJS.I18n.getText('inline.comments.resolve.permission'));
            } else {
                Backbone.trigger("ic:error", AJS.I18n.getText('inline.comments.server.error'));
            }
        },

        showResolvedDialogDiscovery: function() {
            // ResolvedDialogDiscoveryView will auto destroy by itself
            (new ResolvedDialogDiscoveryView()).show();
            Backbone.trigger("ic:resolved:show:recovery");
        },

        onReplyPlaceholderFocus: function(e) {
            if (Utils.getDarkFeatures().RICH_TEXT_EDITOR) {
                // Prevent safari keep the focus on this textarea.
                var isSafari = (navigator.userAgent.indexOf('Safari') !== -1 && navigator.userAgent.indexOf('Chrome') == -1); // Detect Safari only. not Chrome
                isSafari && e.currentTarget.blur();
            }
        }
    });

    return DisplayCommentView;
});
