define('confluence/ic/model/likes',
    [
        'ajs',
        'underscore',
        'backbone',
        'confluence/ic/model/like',
        'confluence/ic/likes/likes-manager'
    ],
    function (
        AJS,
        _,
        Backbone,
        Like,
        LikesManager
    ) {

        "use strict";

        var Likes = Backbone.Collection.extend({
            model: Like,

            url: function () {
                return AJS.contextPath() + "/rest/likes/1.0/content/" + this.contentId + "/likes";
            },

            initialize: function (models, options) {
                _.bindAll(this, "_handleSaveSuccess", "_handleDestroySuccess");
                options = options || {};

                if (options.contentId === undefined) {
                    throw new Error("content ID is required.");
                }

                this.contentId = options.contentId;
                // if the likes array is in the local cache (handled by LikesManager)
                if (LikesManager.getLikes(options.contentId)) {
                    this.reset(LikesManager.getLikes(options.contentId));
                }
                // first load with sidebar open
                this.listenTo(Backbone, "ic:likes-received", this.updateLikes);

                this.bind("add", function (model) {
                    model.save(null, {
                        success: this._handleSaveSuccess,
                        error: this._handleError
                    });
                });
                this.bind("remove", function (model) {
                    // If model.id is null, Backbone will not send a DELETE request. Because Likes do not have an id, we
                    // need to set it before calling destroy(). The model.id must be an empty string to conform to the expectation
                    // of the REST path, which does not expect an id.
                    model.id = "";
                    model.destroy({
                        success: this._handleDestroySuccess,
                        error: this._handleError
                    });
                });
                this.currentUserName = AJS.Meta.get("remote-user");
            },

            updateLikes: function() {
                this.reset(LikesManager.getLikes(this.contentId));
            },

            addUser: function (name) {
                return this.add({ username: name });
            },

            removeUser: function (name) {
                return this.remove(this.getUserByName(name));
            },

            isLikedByUser: function (name) {
                return !!this.getUserByName(name);
            },

            getUserByName: function (name) {
                return this.findWhere({ username: name });
            },

            toggleLike: function() {
                if (this.isLikedByUser(this.currentUserName)) {
                    this.removeUser(this.currentUserName);
                } else {
                    this.addUser(this.currentUserName);
                }
            },

            _handleSaveSuccess: function(model) {
                LikesManager.add(this.contentId, this.currentUserName);
            },

            _handleDestroySuccess: function(model) {
                LikesManager.remove(this.contentId, this.currentUserName);
            },

            _handleError: function(model) {
                Backbone.trigger("ic:error", AJS.I18n.getText('inline.comments.likes.error'));
            }
        });

        return Likes;
    }
);
