define('confluence/ic/app/app',
    [
        'jquery',
        'backbone',
        'ajs',
        'confluence/ic/model/comment',
        'confluence/ic/model/comment-collection',
        'confluence/ic/analytics',
        'exports'
    ],
    function(
        $,
        Backbone,
        AJS,
        Comment,
        CommentCollection,
        Analytics,
        exports
    ) {

    "use strict";

    function init() {
        Analytics.start();

        var firstCaller = true;
        var commentCollection = new CommentCollection();

        function initAsync() {
            downloadJS().done(function (loader) {
                loader.afterSyncCommentCollection();
                loader.displayPermalinkedComment(commentCollection);
            });
        }

        $('body').on('click', '#view-resolved-comments', function(event) {
            event.preventDefault();
            downloadJS().done(function(loader){
                loader.createResolvedCommentListView();
            });
        });

        commentCollection.fetch({cache: false}).done(function(){
            if(commentCollection.getCommentsOnPageCount() > 0){
                initAsync();
            }
            if (commentCollection.getResolvedCount() > 0) {
                downloadJS().done(function(loader){
                    loader.updateResolvedCommentCountInToolsMenu();
                });
            }
        });

        var queryString = window.location.search;
        if (queryString.indexOf('focusedCommentId') !== -1) {
            // Always load resources in case access from comment links
            initAsync();
        }

        //Resource should be loaded after resolving comments
        Backbone.listenTo(Backbone, "ic:resolved", function () {
            initAsync();
        });

        var $viewResolvedCommentsDisplay = $('#view-resolved-comments>span');
        $viewResolvedCommentsDisplay.text(AJS.I18n.getText('inline.comments.resolved.view.menu') +
            AJS.format(' ({0})', commentCollection.getResolvedCount()));

        // start quick-reload plugin integration
        var onShowNewThread = function(e, commentId) {
            var comment = commentCollection.get(commentId);
            var isNew = false;
            if (comment === undefined) { // quick reloading a reply or the current comment
                comment = new Comment({
                    id : commentId
                });
                isNew = true;
            }
            // fetch and merge remaining model data from serve
            comment.fetch({
                success: function(model) {
                    // the highlight reference is usually set in the constructor but because we do not have access
                    // to the markerRef before the model is fetched, we need to call the internal _setHighlights on success
                    model._setHighlights(model.get('markerRef'));
                    if (isNew) {
                        commentCollection.add(model);
                    } else {
                        comment.replies.isFetched = false; // Hack!!!! - to force loading of replies
                    }
                    Backbone.trigger("ic:view", comment, 'quickreload');
                }
            });
        };
        AJS.bind("qr:show-new-thread", function(e, commentId) {
            downloadJS().done(function() {
                onShowNewThread(e, commentId);
            });
        });
        AJS.bind("qr:add-new-highlight", function(e, commentAndUser, deferred) {
            downloadJS().done(function() {
                AJS.trigger("qr:add-new-highlight-text", [commentAndUser, deferred]);
            });
        });
        // end quick-reload plugin integration

        var download;
        var isSidebarLoaded;
        function downloadJS() {
            if (firstCaller) {
                firstCaller = false; //we just need to load this one one time.
                if (download) {
                    return download.promise();
                }
                download = $.Deferred();
                WRM.require('wrc!inline-comments-load-sidebar').then(function () {
                    var Loader = require('confluence/ic/app/loader');
                    Loader.init(commentCollection);
                    download.resolve(Loader);
                });
            }
            isSidebarLoaded = false;
            return download.promise();
        }

        var PLUGIN_KEY = "com.atlassian.confluence.plugins.confluence-inline-comments:create-inline-comment";
        if (Confluence && Confluence.HighlightAction) {
            Confluence.HighlightAction.registerButtonHandler(PLUGIN_KEY, {
                onClick: function(selectionObject) {
                    downloadJS().done(function(loader) {
                        if(isSidebarLoaded == false){
                            isSidebarLoaded = true;
                            loader.loadSidebarOnClick(selectionObject);
                        }
                    });
                },
                shouldDisplay: Confluence.HighlightAction.WORKING_AREA.MAINCONTENT_ONLY
            });
        }
    }

    exports.init = init;
});