package com.atlassian.confluence.plugins.createcontent.upgrade;

import com.atlassian.bandana.BandanaManager;
import com.atlassian.confluence.plugins.createcontent.BlueprintConstants;
import com.atlassian.confluence.plugins.createcontent.BlueprintStateController;
import com.atlassian.confluence.plugins.createcontent.ContentBlueprintManager;
import com.atlassian.confluence.plugins.createcontent.SpaceBandanaContext;
import com.atlassian.confluence.plugins.createcontent.impl.ContentBlueprint;
import com.atlassian.confluence.spaces.Space;
import com.atlassian.confluence.spaces.SpaceManager;
import com.atlassian.plugin.ModuleCompleteKey;
import com.atlassian.plugin.web.WebInterfaceManager;
import com.atlassian.plugin.web.descriptors.WebItemModuleDescriptor;
import com.atlassian.sal.api.message.Message;
import com.atlassian.sal.api.upgrade.PluginUpgradeTask;
import com.google.common.base.Predicate;
import com.google.common.collect.Iterables;
import com.google.common.collect.Sets;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.annotation.Nullable;
import java.util.Collection;
import java.util.List;
import java.util.Set;
import java.util.UUID;

import static com.atlassian.confluence.plugins.createcontent.api.contextproviders.BlueprintContextKeys.BLUEPRINT_MODULE_KEY;

/**
 * Upgrades space-level disabled blueprints related entries in Bandana, so we use UUIDs instead of module complete keys.
 * Entries that cannot be updated, will be deleted. This will happens if a space-level disabled blueprint
 * (which keeps an entry in Bandana) is disabled globally afterwards.
 * <p>
 * Before upgrading, space disabled blueprints were stored in Bandana using their module complete key.
 * Currently we expect to find their UUID.
 */
public class SpaceLevelDisabledBlueprintsUpgradeTask implements PluginUpgradeTask {
    private static final Logger log = LoggerFactory.getLogger(SpaceLevelDisabledBlueprintsUpgradeTask.class);
    private static final String CREATE_DIALOG_CONTENT_SECTION = "system.create.dialog/content";

    private final BandanaManager bandanaManager;

    private final ContentBlueprintManager contentBlueprintManager;

    private final SpaceManager spaceManager;

    private final WebInterfaceManager webInterfaceManager;

    private final BlueprintStateController blueprintStateController;

    public SpaceLevelDisabledBlueprintsUpgradeTask(BandanaManager bandanaManager,
                                                   ContentBlueprintManager contentBlueprintManager, SpaceManager spaceManager,
                                                   WebInterfaceManager webInterfaceManager, BlueprintStateController blueprintStateController) {
        this.bandanaManager = bandanaManager;
        this.contentBlueprintManager = contentBlueprintManager;
        this.spaceManager = spaceManager;
        this.webInterfaceManager = webInterfaceManager;
        this.blueprintStateController = blueprintStateController;
    }

    /**
     * The build number for this upgrade task. Once this upgrade task has run the plugin manager will store this
     * build number against this plugin type.  After this only upgrade tasks with higher build numbers will be run
     */
    @Override
    public int getBuildNumber() {
        return 3;
    }

    @Override
    public String getShortDescription() {
        return "Updates the Bandana entries for space-level disabled blueprints, so they are referenced using UUIDs.";
    }

    @Override
    public Collection<Message> doUpgrade() {
        final List<Space> allSpaces = spaceManager.getAllSpaces();

        // Gets all web-items loaded in the Create Dialog - this will return just the global enabled ones!
        final Iterable<WebItemModuleDescriptor> enabledWebItems = Iterables.filter(
                webInterfaceManager.getItems(CREATE_DIALOG_CONTENT_SECTION),
                new Predicate<WebItemModuleDescriptor>() {
                    @Override
                    public boolean apply(@Nullable WebItemModuleDescriptor webItem) {
                        if (webItem == null)
                            return false;

                        String moduleKey = webItem.getParams().get(BLUEPRINT_MODULE_KEY.key());
                        if (moduleKey == null) {
                            log.warn("Can't find module key for web item {}", webItem);
                            return false;
                        }

                        return true;
                    }
                }
        );

        for (Space space : allSpaces) {
            updateSpaceDisabledBlueprints(space, enabledWebItems);
        }

        return null;
    }

    private void updateSpaceDisabledBlueprints(final Space space, final Iterable<WebItemModuleDescriptor> enabledWebItems) {
        final SpaceBandanaContext bandanaContext = new SpaceBandanaContext(space);

        // Gets all disabled blueprints stored in Bandana
        final Iterable<String> spaceDisabledWebItemModuleCompleteKeys = bandanaManager.getKeys(bandanaContext);

        if (!spaceDisabledWebItemModuleCompleteKeys.iterator().hasNext())
            return;                             // No space-level disabled blueprints, do nothing!

        final Set<UUID> spaceDisabledBlueprintIds = Sets.newHashSet();
        for (final String disabledWebItemModuleCompleteKey : spaceDisabledWebItemModuleCompleteKeys) {
            final ModuleCompleteKey blueprintModuleCompleteKey = findBlueprintModuleCompleteKey(enabledWebItems, disabledWebItemModuleCompleteKey);

            // If content blueprint is found then we add it to the list to be stored in Bandana
            if (blueprintModuleCompleteKey != null) {
                ContentBlueprint contentBlueprint = contentBlueprintManager.getOrCreateCustomBlueprint(blueprintModuleCompleteKey, space);
                spaceDisabledBlueprintIds.add(contentBlueprint.getId());
            }

            // Bandana entry pointing to web-item module complete key is always deleted
            bandanaManager.removeValue(bandanaContext, disabledWebItemModuleCompleteKey);
        }

        blueprintStateController.disableBlueprints(spaceDisabledBlueprintIds, space);
    }

    /**
     * Returns the Blueprint module complete key related to a a create-content dialog web-item.
     * The blueprint will be found if it's enabled globally, otherwise it will return null.
     *
     * @param webItems                 list of web-webItems loaded in the Create content section
     * @param webItemModuleCompleteKey module complete key of the web-item
     * @return the module complete key of the blueprint associated to a web-item if found, otherwise null
     */
    private ModuleCompleteKey findBlueprintModuleCompleteKey(final Iterable<WebItemModuleDescriptor> webItems,
                                                             final String webItemModuleCompleteKey) {
        for (final WebItemModuleDescriptor webItem : webItems) {
            if (webItemModuleCompleteKey.equals(webItem.getCompleteKey())) {
                final String moduleKey = webItem.getParams().get(BLUEPRINT_MODULE_KEY.key());
                return new ModuleCompleteKey(webItem.getPluginKey(), moduleKey);
            }
        }
        return null;
    }

    @Override
    public String getPluginKey() {
        return BlueprintConstants.PLUGIN_KEY;
    }

}
