package com.atlassian.confluence.plugins.createcontent.listeners;

import com.atlassian.confluence.pages.Page;
import com.atlassian.confluence.plugins.createcontent.TemplatePageCreateEvent;
import com.atlassian.confluence.plugins.createcontent.actions.BlueprintManager;
import com.atlassian.confluence.plugins.createcontent.impl.ContentTemplateRef;
import com.atlassian.event.api.EventListener;
import com.google.common.collect.Maps;

import java.util.List;
import java.util.Map;

import static com.atlassian.confluence.plugins.createcontent.actions.DefaultBlueprintContentGenerator.CONTENT_PAGE_TITLE_CONTEXT_KEY;
import static com.atlassian.confluence.plugins.createcontent.actions.DefaultBlueprintContentGenerator.PAGE_TITLE_PREFIX_CONTEXT_KEY;
import static com.atlassian.confluence.plugins.createcontent.actions.DefaultBlueprintContentGenerator.USE_PAGE_TEMPLATE_TITLE_CONTEXT_KEY;
import static com.atlassian.confluence.plugins.createcontent.api.contextproviders.BlueprintContextKeys.NO_PAGE_TITLE_PREFIX;

/**
 * Creates child Templates of a page after it is created.
 *
 * @since 1.6
 */
public class TemplateChildCreatorListener {
    private final BlueprintManager blueprintManager;

    public TemplateChildCreatorListener(final BlueprintManager blueprintManager) {
        this.blueprintManager = blueprintManager;
    }

    @EventListener
    public void onCreateEvent(final TemplatePageCreateEvent event) {
        ContentTemplateRef templateRef = event.getTemplateRef();
        List<ContentTemplateRef> children = templateRef.getChildren();
        if (children.isEmpty())
            return;

        Page parentPage = event.getPage();
        String parentTitle = parentPage.getTitle();

        Map<String, Object> originalContext = event.getContext();

        boolean addPageTitlePrefix = !"true".equals(originalContext.get(NO_PAGE_TITLE_PREFIX.key()));

        for (ContentTemplateRef child : children) {
            Map<String, Object> childContext = Maps.newHashMap(originalContext);

            // CONTENT_PAGE_TITLE_CONTEXT_KEY will be the parent-page key; unset it.
            childContext.remove(CONTENT_PAGE_TITLE_CONTEXT_KEY);
            childContext.put(USE_PAGE_TEMPLATE_TITLE_CONTEXT_KEY, true);
            if (addPageTitlePrefix) {
                childContext.put(PAGE_TITLE_PREFIX_CONTEXT_KEY, parentTitle);
            }

            // TODO - expose method in this manager to create page from template.
            blueprintManager.createPageFromTemplate(child, event.getUser(), parentPage.getSpace(),
                    parentPage, childContext, event.getSaveContext());
        }
    }
}
