package com.atlassian.confluence.plugins.createcontent.impl;

import com.atlassian.confluence.pages.templates.PageTemplate;
import com.atlassian.confluence.pages.templates.PageTemplateManager;
import com.atlassian.confluence.plugins.createcontent.extensions.BlueprintModuleDescriptor;
import com.atlassian.confluence.plugins.createcontent.rest.IconUrlProvider;
import com.atlassian.confluence.plugins.createcontent.rest.PageTemplateWebItemService;
import com.atlassian.confluence.plugins.createcontent.rest.entities.CreateDialogWebItemEntity;
import com.atlassian.confluence.security.PermissionManager;
import com.atlassian.confluence.spaces.Space;
import com.atlassian.confluence.user.ConfluenceUser;
import com.atlassian.plugin.ModuleCompleteKey;
import com.atlassian.plugin.PluginAccessor;

import java.util.Collection;
import java.util.List;
import java.util.Set;

import static com.atlassian.confluence.security.Permission.VIEW;
import static com.atlassian.confluence.security.PermissionManager.TARGET_APPLICATION;
import static com.google.common.collect.Lists.newArrayList;
import static com.google.common.collect.Lists.newLinkedList;
import static com.google.common.collect.Sets.newHashSet;
import static org.apache.commons.lang3.StringUtils.defaultIfBlank;

public class DefaultPageTemplateWebItemService implements PageTemplateWebItemService {
    private final PermissionManager permissionManager;
    private final PageTemplateManager pageTemplateManager;
    private final PluginAccessor pluginAccessor;
    private final IconUrlProvider iconUrlProvider;

    public DefaultPageTemplateWebItemService(PermissionManager permissionManager,
                                             PageTemplateManager pageTemplateManager, PluginAccessor pluginAccessor, IconUrlProvider iconUrlProvider) {
        this.permissionManager = permissionManager;
        this.pageTemplateManager = pageTemplateManager;
        this.pluginAccessor = pluginAccessor;
        this.iconUrlProvider = iconUrlProvider;
    }

    @Override
    public List<CreateDialogWebItemEntity> getPageTemplateItems(Space space, ConfluenceUser user) {
        final List<CreateDialogWebItemEntity> entities = newArrayList();
        final List<PageTemplate> pageTemplates = newLinkedList();

        if (space != null && permissionManager.hasPermission(user, VIEW, space)) {
            pageTemplates.addAll(space.getPageTemplates());
        }
        if (permissionManager.hasPermission(user, VIEW, TARGET_APPLICATION)) {
            pageTemplates.addAll(pageTemplateManager.getGlobalPageTemplates());
        }
        final Set<String> blueprintTemplateKeys = getBlueprintTemplateKeys();
        for (final PageTemplate pageTemplate : pageTemplates) {
            if (shouldHideTemplate(blueprintTemplateKeys, pageTemplate))
                continue;

            final String description = defaultIfBlank(pageTemplate.getDescription(), "");
            final String iconURL = iconUrlProvider.getDefaultIconUrl();
            final String templateId = String.valueOf(pageTemplate.getId());

            entities.add(new CreateDialogWebItemEntity(pageTemplate.getName(), description, "icon-content-template-large",
                    iconURL, templateId));
        }

        return entities;
    }

    // Don't display system templates, or templates associated with a blueprint
    private boolean shouldHideTemplate(final Set<String> blueprintTemplateKeys, final PageTemplate pageTemplate) {
        if (pageTemplate.getReferencingModuleCompleteKey() != null)
            // Owned by a plugin Blueprint - don't show as a standalone template.
            return true;

        final ModuleCompleteKey templateKey = pageTemplate.getModuleCompleteKey();
        final boolean isBlueprintTemplateKey = templateKey != null &&
                blueprintTemplateKeys.contains(templateKey.getCompleteKey());

        if (isBlueprintTemplateKey)
            return true;

        final String pluginKey = pageTemplate.getPluginKey();
        return (pluginKey != null && pluginAccessor.isSystemPlugin(pluginKey));
    }

    // FIXME - this method will be insufficient once we allow for user-generated Blueprints that point to user-generated
    // templates. We'll need to get all ContentBlueprints for the space (including globals) and check the
    // ContentTemplateRefs for those blueprints - PageTemplates matching the refs will be ignored.
    private Set<String> getBlueprintTemplateKeys() {
        final Collection<BlueprintModuleDescriptor> moduleDescriptors = pluginAccessor.getEnabledModuleDescriptorsByClass(
                BlueprintModuleDescriptor.class);

        final Set<String> blueprintTemplateKeys = newHashSet();
        for (final BlueprintModuleDescriptor moduleDescriptor : moduleDescriptors) {
            for (ModuleCompleteKey templateKey : moduleDescriptor.getContentTemplates()) {
                blueprintTemplateKeys.add(templateKey.getCompleteKey());
            }
            if (moduleDescriptor.getIndexTemplate() != null) {
                blueprintTemplateKeys.add(moduleDescriptor.getIndexTemplate().getCompleteKey());
            }
        }

        return blueprintTemplateKeys;
    }
}
