package com.atlassian.confluence.plugins.createcontent.impl;

import com.atlassian.activeobjects.external.ActiveObjects;
import com.atlassian.confluence.languages.LocaleManager;
import com.atlassian.confluence.plugins.createcontent.ContentBlueprintManager;
import com.atlassian.confluence.plugins.createcontent.activeobjects.ContentBlueprintAo;
import com.atlassian.confluence.plugins.createcontent.activeobjects.ContentTemplateRefAo;
import com.atlassian.confluence.plugins.createcontent.api.exceptions.ResourceErrorType;
import com.atlassian.confluence.plugins.createcontent.concurrent.LazyInsertExecutor;
import com.atlassian.confluence.plugins.createcontent.concurrent.LazyInserter;
import com.atlassian.confluence.plugins.createcontent.exceptions.BlueprintPluginNotFoundException;
import com.atlassian.confluence.plugins.createcontent.exceptions.ModuleNotBlueprintException;
import com.atlassian.confluence.plugins.createcontent.extensions.BlueprintModuleDescriptor;
import com.atlassian.confluence.plugins.createcontent.extensions.ContentTemplateModuleDescriptor;
import com.atlassian.confluence.spaces.Space;
import com.atlassian.confluence.util.i18n.I18NBeanFactory;
import com.atlassian.plugin.ModuleCompleteKey;
import com.atlassian.plugin.ModuleDescriptor;
import com.atlassian.plugin.PluginAccessor;
import com.atlassian.sal.api.transaction.TransactionCallback;
import com.google.common.collect.Lists;
import net.java.ao.Query;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.Arrays;
import java.util.List;
import java.util.UUID;

import static com.atlassian.confluence.plugins.createcontent.BlueprintConstants.BLANK_PAGE_BLUEPRINT;
import static com.atlassian.confluence.plugins.createcontent.BlueprintConstants.BLOG_POST_BLUEPRINT;
import static com.atlassian.confluence.plugins.createcontent.BlueprintConstants.MODULE_KEY_BLANK_PAGE;
import static com.atlassian.confluence.plugins.createcontent.BlueprintConstants.MODULE_KEY_BLOG_POST;
import static com.atlassian.confluence.plugins.createcontent.impl.ActiveObjectsUtils.createWithUuid;

/**
 * ActiveObjects-backed manager for saving and retrieving {@link ContentBlueprint}s.
 *
 * @since 1.6
 */
public class DefaultContentBlueprintManager extends AbstractAoManager<ContentBlueprint, ContentBlueprintAo> implements ContentBlueprintManager {
    private static final Logger log = LoggerFactory.getLogger(DefaultContentBlueprintManager.class);

    private final PluginAccessor pluginAccessor;
    private final DefaultContentTemplateRefManager contentTemplateRefManager;
    private final I18NBeanFactory i18NBeanFactory;
    private final LocaleManager localeManager;
    private final LazyInsertExecutor lazyInsertExecutor;

    public DefaultContentBlueprintManager(@Nonnull ActiveObjects activeObjects,
                                          @Nonnull PluginAccessor pluginAccessor,
                                          @Nonnull DefaultContentTemplateRefManager contentTemplateRefManager,
                                          @Nonnull I18NBeanFactory i18NBeanFactory,
                                          @Nonnull LocaleManager localeManager,
                                          @Nonnull LazyInsertExecutor lazyInsertExecutor) {
        super(activeObjects, ContentBlueprintAo.class);

        this.pluginAccessor = pluginAccessor;
        this.contentTemplateRefManager = contentTemplateRefManager;
        this.i18NBeanFactory = i18NBeanFactory;
        this.localeManager = localeManager;
        this.lazyInsertExecutor = lazyInsertExecutor;
    }

    @Nullable
    private ContentBlueprint getOrCreateClonedBlueprint(@Nonnull ModuleCompleteKey moduleCompleteKey) {
        try {
            // Gets the blueprint module descriptor, enabled or not
            BlueprintModuleDescriptor pluginBlueprint = getBlueprintDescriptor(moduleCompleteKey.getCompleteKey());

            // Gets blueprint - in case it has been already cloned in AO, otherwise creates one
            return getOrCreateClonedBlueprint(moduleCompleteKey, pluginBlueprint);
        } catch (ModuleNotBlueprintException e) {
            return null;
        }
    }

    @Nullable
    @Override
    public ContentBlueprint getPluginBlueprint(@Nonnull ModuleCompleteKey moduleCompleteKey) {
        // Special case - return Blank page and Blog post blueprints if the moduleCompleteKey matches
        if (moduleCompleteKey.getCompleteKey().equals(MODULE_KEY_BLANK_PAGE))
            return BLANK_PAGE_BLUEPRINT;
        else if (moduleCompleteKey.getCompleteKey().equals(MODULE_KEY_BLOG_POST))
            return BLOG_POST_BLUEPRINT;

        try {
            // 1. Check that the plugin is still enabled
            BlueprintModuleDescriptor pluginBlueprint = getEnabledBlueprintDescriptor(moduleCompleteKey.getCompleteKey());

            // Gets blueprint - in case it has been already cloned in AO, otherwise creates one
            return getOrCreateClonedBlueprint(moduleCompleteKey, pluginBlueprint);
        } catch (ModuleNotBlueprintException e) {
            // return null if the module is not a blueprint
            return null;
        }
    }

    private ContentBlueprint getOrCreateClonedBlueprint(@Nonnull final ModuleCompleteKey moduleCompleteKey,
                                                        @Nullable final BlueprintModuleDescriptor pluginBlueprint) {
        if (pluginBlueprint == null)
            return null;

        // CONFDEV-19787, CONF-33299: If we don't synchronize, we can end up having multiple clones in the DB
        return lazyInsertExecutor.lazyInsertAndRead(new LazyInserter<ContentBlueprint>() {
            @Override
            public ContentBlueprint read() {
                // Find blueprint in case it has been already cloned in AO
                ContentBlueprintAo blueprintAo = getContentBlueprint(moduleCompleteKey, null, true);
                return blueprintAo != null ? build(blueprintAo) : null;
            }

            @Override
            public ContentBlueprint insert() {
                // Store blueprint as it has not been cloned yet in AO
                return storePluginBlueprintClone(pluginBlueprint);
            }
        }, "cc:getOrCreateClonedBlueprint:" + moduleCompleteKey.getCompleteKey());
    }

    @Nonnull
    @Override
    public List<ContentBlueprint> getAll(Space space) {
        if (space == null)
            return super.getAll("SPACE_KEY IS NULL");

        return super.getAll("SPACE_KEY = ?", space.getKey());
    }

    @Nullable
    @Override
    public ContentBlueprint getPluginBackedContentBlueprint(ModuleCompleteKey moduleCompleteKey, String spaceKey) {
        // Looks up for a space-level override of the content blueprint
        ContentBlueprintAo blueprintAo = null;
        if (spaceKey != null) {
            blueprintAo = getContentBlueprint(moduleCompleteKey, spaceKey, false);
        }
        if (blueprintAo == null) {
            // No space level, then look for a global-level Custom Bp
            // This call and the one above could be joined into one DB call...
            blueprintAo = getContentBlueprint(moduleCompleteKey, null, false);
        }

        if (blueprintAo != null)
            return build(blueprintAo);

        // No space-level Bp, return the global version
        return getPluginBlueprint(moduleCompleteKey);
    }

    @Nonnull
    @Override
    protected ContentBlueprintAo internalCreateAo(@Nonnull final ContentBlueprint original) {
        ContentBlueprintAo ao = createWithUuid(activeObjects, ContentBlueprintAo.class);

        ContentTemplateRef indexPageTemplateRef = original.getIndexPageTemplateRef();
        if (indexPageTemplateRef != null) {
            ContentTemplateRefAo childAo = contentTemplateRefManager.internalCreateAo(indexPageTemplateRef);
            childAo.setContentBlueprintIndexParent(ao);
            childAo.save();
        }

        for (ContentTemplateRef moduleCompleteKey : original.getContentTemplateRefs()) {
            final ContentTemplateRefAo childAo = contentTemplateRefManager.internalCreateAo(moduleCompleteKey);
            childAo.setContentBlueprintParent(ao);
            childAo.save();
        }

        copyPropertiesIntoAo(ao, original, true);
        ao.save();

        return ao;
    }

    @Nonnull
    @Override
    protected ContentBlueprintAo internalUpdateAo(@Nonnull final ContentBlueprint object) {
        ContentBlueprintAo ao = internalGetAoById(object.getId());
        if (ao == null) {
            String error = String.format("Blueprint with UUID %s not found", object.getId());
            throw new IllegalStateException(error);
        }

        ContentTemplateRef indexPageTemplateRef = object.getIndexPageTemplateRef();
        if (indexPageTemplateRef != null) {
            ContentTemplateRefAo childAo = contentTemplateRefManager.internalUpdateAo(indexPageTemplateRef);
            childAo.setContentBlueprintIndexParent(ao);
            childAo.save();
        }
        for (ContentTemplateRef contentTemplateRef : object.getContentTemplateRefs()) {
            ContentTemplateRefAo childAo = contentTemplateRefManager.internalUpdateAo(contentTemplateRef);
            childAo.setContentBlueprintParent(ao);
            childAo.save();
        }

        copyPropertiesIntoAo(ao, object, false);
        ao.save();

        return ao;
    }

    private void copyPropertiesIntoAo(@Nonnull ContentBlueprintAo ao, @Nonnull ContentBlueprint original, boolean isCreate) {
        if (isCreate) {
            ao.setPluginModuleKey(original.getModuleCompleteKey());
        }
        ao.setPluginClone(original.isPluginClone());
        ao.setI18nNameKey(original.getI18nNameKey());
        ao.setSpaceKey(original.getSpaceKey());
        ao.setCreateResult(original.getCreateResult());
        ao.setHowToUseTemplate(original.getHowToUseTemplate());
        ao.setIndexKey(original.getIndexKey());
        ao.setIndexTitleI18nKey(original.getIndexTitleI18nKey());
    }

    @Nonnull
    @Override
    public ContentBlueprint getOrCreateCustomBlueprint(@Nonnull final ModuleCompleteKey moduleCompleteKey, @Nullable final Space space) {
        final String spaceKey = getSpaceKey(space);
        final String spaceSuffix = spaceKey != null ? (":" + spaceKey) : "";

        // CONFDEV-19787, CONF-33299: If we don't synchronize, we can end up having multiple clones in the DB
        return lazyInsertExecutor.lazyInsertAndRead(new LazyInserter<ContentBlueprint>() {
            @Override
            public ContentBlueprint read() {
                ContentBlueprintAo ao = getContentBlueprint(moduleCompleteKey, spaceKey, false);
                return ao != null ? build(ao) : null;
            }

            @Override
            public ContentBlueprint insert() {
                return activeObjects.executeInTransaction(new TransactionCallback<ContentBlueprint>() {
                    @Override
                    public ContentBlueprint doInTransaction() {
                        ContentBlueprint pluginBlueprintClone = getOrCreateClonedBlueprint(moduleCompleteKey);
                        if (pluginBlueprintClone == null) {
                            throw new BlueprintPluginNotFoundException("Tried to create a clone of a disabled / " +
                                    "uninstalled / non existing blueprint with module " + moduleCompleteKey, ResourceErrorType.NOT_FOUND_BLUEPRINT, moduleCompleteKey.getCompleteKey());
                        }
                        pluginBlueprintClone.setId(null);
                        pluginBlueprintClone.setPluginClone(false);
                        pluginBlueprintClone.setSpaceKey(spaceKey);

                        // Set templates pluginClone state
                        pluginBlueprintClone.getIndexPageTemplateRef().setPluginClone(false);
                        List<ContentTemplateRef> contentTemplateRefs = pluginBlueprintClone.getContentTemplateRefs();
                        for (ContentTemplateRef contentTemplateRef : contentTemplateRefs) {
                            contentTemplateRef.setPluginClone(false);
                        }

                        // TODO - maybe could NOT create here and then update: could just return an "unattached" ContentBP.
                        return create(pluginBlueprintClone);
                    }
                });
            }
        }, "cc:getOrCreateCustomBlueprint:" + moduleCompleteKey.getCompleteKey() + spaceSuffix);
    }

    /*
     * Returns a ContentBlueprintAo using its module module key.
     * We may be looking for the cloned blueprint (isPluginClone set to true), of an user-override (isPluginClone set to false).
     * In case of user-defined overrides (created when editing a blueprint template), space-level will have space not null,
     * otherwise it'd a global one.
     */
    @Nullable
    private ContentBlueprintAo getContentBlueprint(@Nonnull final ModuleCompleteKey moduleCompleteKey, @Nullable final String spaceKey, final boolean isPluginClone) {
        return activeObjects.executeInTransaction(new TransactionCallback<ContentBlueprintAo>() {
            @Override
            public ContentBlueprintAo doInTransaction() {
                String completeKey = moduleCompleteKey.getCompleteKey();
                Query query = Query.select();
                if (spaceKey == null) {
                    query.setWhereClause("PLUGIN_MODULE_KEY = ? AND SPACE_KEY IS NULL AND PLUGIN_CLONE = ?");
                    query.setWhereParams(new Object[]{completeKey, isPluginClone});
                } else {
                    query.setWhereClause("PLUGIN_MODULE_KEY = ? AND SPACE_KEY = ? AND PLUGIN_CLONE = ?");
                    query.setWhereParams(new Object[]{completeKey, spaceKey, isPluginClone});
                }
                query.order("ID");

                List<ContentBlueprintAo> aos = Arrays.asList(activeObjects.find(ContentBlueprintAo.class, query));

                if (aos.isEmpty())
                    return null;

                if (aos.size() > 1) {
                    // Found more than one Bp for key and space - this is very bad.
                    log.warn(String.format("Should only be one Blueprint with space %s and module key %s", spaceKey,
                            completeKey));
                }

                return aos.get(0);
            }
        });
    }

    @Nullable
    private String getSpaceKey(@Nullable Space space) {
        if (space != null) {
            final String spaceKey = space.getKey().trim();
            return spaceKey.isEmpty() ? null : spaceKey;
        }
        return null;
    }

    @Override
    protected void internalDeleteAo(@Nonnull final ContentBlueprintAo ao) {
        // Delete the associated index template & other templates as well, because now isn't possible to share it with another blueprint
        ContentTemplateRefAo indexTemplateRef = ao.getIndexTemplateRef();
        if (indexTemplateRef != null) {
            contentTemplateRefManager.internalDeleteAo(indexTemplateRef);
            activeObjects.delete(indexTemplateRef);
        }
        ContentTemplateRefAo[] contentTemplates = ao.getContentTemplates();
        for (ContentTemplateRefAo contentTemplate : contentTemplates) {
            contentTemplateRefManager.internalDeleteAo(contentTemplate);
        }
        activeObjects.delete(contentTemplates);
    }

    @Nullable
    private BlueprintModuleDescriptor getEnabledBlueprintDescriptor(@Nonnull final String blueprintKey) {
        return castIfModuleIsBlueprint(blueprintKey, pluginAccessor.getEnabledPluginModule(blueprintKey));
    }

    @Nullable
    private BlueprintModuleDescriptor getBlueprintDescriptor(@Nonnull final String blueprintKey) {
        return castIfModuleIsBlueprint(blueprintKey, pluginAccessor.getPluginModule(blueprintKey));
    }

    private BlueprintModuleDescriptor castIfModuleIsBlueprint(@Nonnull final String blueprintKey,
                                                              @Nullable final ModuleDescriptor<?> pluginModule) {
        if (pluginModule == null)
            return null;

        if (pluginModule instanceof BlueprintModuleDescriptor)
            return (BlueprintModuleDescriptor) pluginModule;

        throw new ModuleNotBlueprintException(blueprintKey, pluginModule, ResourceErrorType.INVALID_MODULE, blueprintKey);
    }

    // Every Plugin Blueprint should be copied into AO before being used, so that it can be referred to via an ID,
    // edited, etc.
    @Nonnull
    private ContentBlueprint storePluginBlueprintClone(@Nonnull final BlueprintModuleDescriptor pluginBlueprint) {
        // FIXME: Put transaction on the calling method, not here
        ContentBlueprintAo ao = activeObjects.executeInTransaction(new TransactionCallback<ContentBlueprintAo>() {
            @Override
            public ContentBlueprintAo doInTransaction() {
                ContentBlueprintAo newBp = createWithUuid(activeObjects, ContentBlueprintAo.class);

                String blueprintKey = pluginBlueprint.getCompleteKey();
                newBp.setPluginModuleKey(blueprintKey);
                newBp.setPluginClone(true);
                String i18nNameKey = getI18nNameKey(pluginBlueprint);
                newBp.setI18nNameKey(i18nNameKey);
                newBp.setCreateResult(pluginBlueprint.getCreateResult());
                newBp.setHowToUseTemplate(pluginBlueprint.getHowToUseTemplate());
                newBp.setIndexKey(pluginBlueprint.getIndexKey());
                newBp.setIndexTitleI18nKey(pluginBlueprint.getIndexTitleI18nKey());
                newBp.setIndexDisabled(pluginBlueprint.isIndexDisabled());

                newBp.save();

                final ModuleCompleteKey indexTemplateKey = pluginBlueprint.getIndexTemplate();
                if (indexTemplateKey != null) {
                    final ContentTemplateRefAo indexContentTemplateRef = createContentTemplateRefAo(indexTemplateKey);
                    indexContentTemplateRef.setContentBlueprintIndexParent(newBp);
                    indexContentTemplateRef.save();
                }

                for (ModuleCompleteKey moduleCompleteKey : pluginBlueprint.getContentTemplates()) {
                    if (moduleCompleteKey != null) {
                        final ContentTemplateRefAo ao = createContentTemplateRefAo(moduleCompleteKey);
                        ao.setContentBlueprintParent(newBp);
                        ao.save();
                    }
                }

                return newBp;
            }
        });

        return build(ao);
    }

    String getI18nNameKey(@Nonnull final BlueprintModuleDescriptor pluginBlueprint) {
        String i18nNameKey = pluginBlueprint.getI18nNameKey();
        if (StringUtils.isBlank(i18nNameKey)) {
            log.warn("i18n-name-key must be specified for: " + pluginBlueprint.getCompleteKey());
            i18nNameKey = pluginBlueprint.getName();
            if (StringUtils.isBlank(i18nNameKey)) {
                i18nNameKey = pluginBlueprint.getKey();
            }
        }

        return i18nNameKey;
    }

    @Nonnull
    ContentTemplateRefAo createContentTemplateRefAo(@Nonnull ModuleCompleteKey indexTemplateKey) {
        String moduleCompleteKey = indexTemplateKey.getCompleteKey();
        ContentTemplateModuleDescriptor contentTemplateDescriptor =
                (ContentTemplateModuleDescriptor) pluginAccessor.getEnabledPluginModule(moduleCompleteKey);

        if (contentTemplateDescriptor == null) {
            throw new BlueprintPluginNotFoundException("The content-template module descriptor with key '" +
                    moduleCompleteKey + "' was not found", ResourceErrorType.NOT_FOUND_CONTENT_TEMPLATE, moduleCompleteKey);
        }

        String i18nNameKey = getI18nNameKey(contentTemplateDescriptor);
        return contentTemplateRefManager.createAo(new ContentTemplateRef(null, 0, moduleCompleteKey, i18nNameKey, true, null));
    }

    private String getI18nNameKey(@Nonnull ContentTemplateModuleDescriptor contentTemplateDescriptor) {
        String i18nNameKey = contentTemplateDescriptor.getI18nNameKey();
        if (StringUtils.isBlank(i18nNameKey)) {
            log.warn("i18n-name-key must be specified for: " + contentTemplateDescriptor.getCompleteKey());
            i18nNameKey = contentTemplateDescriptor.getKey();
        }
        return i18nNameKey;
    }

    @Nonnull
    protected ContentBlueprint build(@Nonnull ContentBlueprintAo ao) {
        ContentBlueprint contentBlueprint = new ContentBlueprint();
        contentBlueprint.setId(UUID.fromString(ao.getUuid()));
        final String pluginModuleKey = ao.getPluginModuleKey();
        contentBlueprint.setModuleCompleteKey(pluginModuleKey);
        contentBlueprint.setI18nNameKey(ao.getI18nNameKey());
        contentBlueprint.setPluginClone(ao.isPluginClone());
        contentBlueprint.setSpaceKey(ao.getSpaceKey());
        contentBlueprint.setIndexKey(ao.getIndexKey());
        contentBlueprint.setIndexPageTemplateRef(convertAo(ao.getIndexTemplateRef(), contentBlueprint));
        contentBlueprint.setCreateResult(ao.getCreateResult());
        contentBlueprint.setHowToUseTemplate(ao.getHowToUseTemplate());
        contentBlueprint.setIndexTitleI18nKey(ao.getIndexTitleI18nKey());
        contentBlueprint.setIndexDisabled(ao.isIndexDisabled());
        // TODO: This retrieves ALL the children! Danger Will Robinson!
        contentBlueprint.setContentTemplateRefs(convertAos(ao.getContentTemplates(), contentBlueprint));
        // TODO - dialog wizard - should come directly from the respective Plugin Blueprint if there is one

        if (StringUtils.isNotBlank(pluginModuleKey)) {
            final BlueprintModuleDescriptor blueprintDescriptor = getEnabledBlueprintDescriptor(pluginModuleKey);
            if (blueprintDescriptor != null) {
                contentBlueprint.setDialogWizard(blueprintDescriptor.getDialogWizard());
            }
        }

        return contentBlueprint;
    }

    @Nonnull
    private List<ContentTemplateRef> convertAos(@Nonnull ContentTemplateRefAo[] aos, @Nonnull ContentBlueprint parent) {
        List<ContentTemplateRef> result = Lists.newArrayList();
        for (ContentTemplateRefAo ao : aos) {
            // FIXME: This is to get around a "bug?" of AO when, on ManyToMany relationships, the list we obtain contains one "empty" object (ID:0, rest of fields as null)
            if (ao.getID() != 0) {
                result.add(convertAo(ao, parent));
            }
        }
        return result;
    }

    @Nullable
    private ContentTemplateRef convertAo(@Nullable ContentTemplateRefAo ao, @Nonnull ContentBlueprint parent) {
        if (ao == null)
            return null;
        return new ContentTemplateRef(UUID.fromString(ao.getUuid()), ao.getTemplateId(),
                ao.getPluginModuleKey(), ao.getI18nNameKey(), ao.isPluginClone(), parent);
    }
}
