package com.atlassian.confluence.plugins.createcontent.factory;

import com.atlassian.confluence.api.impl.service.content.finder.AbstractFinder;
import com.atlassian.confluence.api.model.Expansion;
import com.atlassian.confluence.api.model.content.Space;
import com.atlassian.confluence.api.model.content.template.ContentTemplate;
import com.atlassian.confluence.api.model.content.template.ContentTemplateId;
import com.atlassian.confluence.api.model.content.template.ContentTemplateType;
import com.atlassian.confluence.api.model.pagination.PageRequest;
import com.atlassian.confluence.api.model.pagination.PageResponse;
import com.atlassian.confluence.api.model.pagination.SimplePageRequest;
import com.atlassian.confluence.api.service.content.template.ContentTemplateService;
import com.atlassian.confluence.api.service.finder.SingleFetcher;
import com.atlassian.confluence.spring.transaction.interceptor.TransactionalHostContextAccessor;
import com.atlassian.fugue.Iterables;
import com.atlassian.fugue.Option;

import static com.google.common.base.Preconditions.checkNotNull;

/**
 * Makes Template Finder instance
 *
 * @since 6.0
 */
public class TemplateFinderFactory implements FinderFactory {
    private final TransactionalHostContextAccessor hostContextAccessor;

    public TemplateFinderFactory(TransactionalHostContextAccessor hostContextAccessor) {
        this.hostContextAccessor = hostContextAccessor;
    }

    @Override
    public ContentTemplateService.TemplateFinder createFinder(ContentTemplateService service, Expansion... expansions) {
        TemplateFinderImpl rawFinder = new TemplateFinderImpl(service, expansions);
        return new TransactionWrappingContentFinder(rawFinder, hostContextAccessor);
    }

    private class TemplateFinderImpl extends AbstractFinder<ContentTemplate> implements ContentTemplateService.TemplateFinder {
        private final ContentTemplateService contentTemplateService;
        private Option<Space> space = Option.none();
        private ContentTemplateId contentTemplateId;
        private ContentTemplateType contentTemplateType;

        public TemplateFinderImpl(ContentTemplateService contentTemplateService, Expansion... expansions) {
            super(expansions);
            this.contentTemplateService = contentTemplateService;
        }

        @Override
        public SingleFetcher<ContentTemplate> withId(ContentTemplateId contentTemplateId) {
            checkNotNull(contentTemplateId);
            this.contentTemplateId = contentTemplateId;
            return this;
        }

        public ContentTemplateService.ParameterTemplateFinder withType(ContentTemplateType contentTemplateType) {
            checkNotNull(contentTemplateType);
            this.contentTemplateType = contentTemplateType;
            return this;
        }

        @Override
        public ContentTemplateService.ParameterTemplateFinder withSpace(Space space) {
            checkNotNull(space);
            this.space = Option.some(space);
            return this;
        }

        @Override
        public PageResponse<ContentTemplate> fetchMany(PageRequest pageRequest) {
            return contentTemplateService.getTemplates(contentTemplateType, space, pageRequest, expansions);
        }

        @Override
        public Option<ContentTemplate> fetchOne() {
            if (contentTemplateId != null) {
                return Option.option(contentTemplateService.getTemplate(this.contentTemplateId, expansions));
            }

            //yield the first template found
            ContentTemplate firstFound = Iterables.first(contentTemplateService.getTemplates(contentTemplateType, space, SimplePageRequest.ONE, expansions)).getOrNull();
            return Option.option(firstFound);
        }
    }

    private static class TransactionWrappingContentFinder extends TransactionWrappingFinder<ContentTemplate>
            implements ContentTemplateService.TemplateFinder {
        private final TemplateFinderImpl delegate;

        public TransactionWrappingContentFinder(TemplateFinderImpl delegate, TransactionalHostContextAccessor hostContextAccessor) {
            super(delegate, delegate, hostContextAccessor);
            this.delegate = delegate;
        }

        @Override
        public SingleFetcher<ContentTemplate> withId(ContentTemplateId contentTemplateId) {
            delegate.withId(contentTemplateId);
            return this;
        }

        public ContentTemplateService.ParameterTemplateFinder withType(ContentTemplateType contentTemplateType) {
            delegate.withType(contentTemplateType);
            return this;
        }

        @Override
        public ContentTemplateService.ParameterTemplateFinder withSpace(Space space) {
            delegate.withSpace(space);
            return this;
        }
    }
}
