package com.atlassian.confluence.plugins.createcontent.actions;

import com.atlassian.confluence.core.ContentPropertyManager;
import com.atlassian.confluence.core.DefaultSaveContext;
import com.atlassian.confluence.core.SaveContext;
import com.atlassian.confluence.labels.Label;
import com.atlassian.confluence.labels.LabelManager;
import com.atlassian.confluence.labels.Labelable;
import com.atlassian.confluence.pages.AbstractPage;
import com.atlassian.confluence.pages.Page;
import com.atlassian.confluence.pages.PageManager;
import com.atlassian.confluence.plugins.createcontent.TemplatePageCreateEvent;
import com.atlassian.confluence.plugins.createcontent.api.events.BlueprintPageCreateEvent;
import com.atlassian.confluence.plugins.createcontent.api.services.ContentBlueprintService;
import com.atlassian.confluence.plugins.createcontent.extensions.BlueprintDescriptor;
import com.atlassian.confluence.plugins.createcontent.extensions.BlueprintModuleDescriptor;
import com.atlassian.confluence.plugins.createcontent.impl.BlueprintLock;
import com.atlassian.confluence.plugins.createcontent.impl.ContentBlueprint;
import com.atlassian.confluence.plugins.createcontent.impl.ContentTemplateRef;
import com.atlassian.confluence.plugins.createcontent.services.BlueprintResolver;
import com.atlassian.confluence.plugins.ia.service.SidebarLinkService;
import com.atlassian.confluence.rpc.NotPermittedException;
import com.atlassian.confluence.spaces.Space;
import com.atlassian.confluence.user.ConfluenceUser;
import com.atlassian.event.api.EventPublisher;
import com.atlassian.plugin.ModuleCompleteKey;
import com.atlassian.plugin.PluginAccessor;
import com.atlassian.plugin.web.descriptors.WebItemModuleDescriptor;
import com.atlassian.sal.api.message.I18nResolver;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;

import java.util.Map;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;

import static com.atlassian.confluence.plugins.createcontent.actions.DefaultBlueprintContentGenerator.CONTENT_TEMPLATE_REF_ID_CONTEXT_KEY;
import static com.google.common.base.Preconditions.checkNotNull;
import static org.apache.commons.lang3.StringUtils.isBlank;
import static org.apache.commons.lang3.StringUtils.isNotBlank;
import static org.slf4j.LoggerFactory.getLogger;

/**
 * Reduces friction between the blueprint actions and the other managers.
 *
 * @since 5.0
 */
public class DefaultBlueprintManager implements BlueprintManager {
    private static final Logger log = getLogger(DefaultBlueprintManager.class);

    private final ConcurrentHashMap<BlueprintLock, BlueprintLock> getOrCreateLocks = new ConcurrentHashMap<BlueprintLock, BlueprintLock>();

    static final String INDEX_PAGE_LABEL = "blueprint-index-page";

    private static final String BLUEPRINT_KEY = "blueprintModuleKey";

    private final PluginAccessor pluginAccessor;
    private final SidebarLinkService sidebarLinkService;
    private final I18nResolver i18nResolver;
    private final BlueprintContentGenerator contentGenerator;
    private final ContentPropertyManager contentPropertyManager;
    private final IndexPageManager indexPageManager;
    private final LabelManager labelManager;
    private final PageManager pageManager;
    private final EventPublisher eventPublisher;
    private final BlueprintResolver resolver;

    public DefaultBlueprintManager(PluginAccessor pluginAccessor,
                                   SidebarLinkService sidebarLinkService, I18nResolver i18nResolver,
                                   BlueprintContentGenerator contentGenerator,
                                   final ContentPropertyManager contentPropertyManager,
                                   IndexPageManager indexPageManager,
                                   LabelManager labelManager, PageManager pageManager, EventPublisher eventPublisher, BlueprintResolver resolver)

    {
        this.pluginAccessor = pluginAccessor;
        this.i18nResolver = i18nResolver;
        this.sidebarLinkService = sidebarLinkService;
        this.contentGenerator = contentGenerator;
        this.contentPropertyManager = contentPropertyManager;
        this.indexPageManager = indexPageManager;
        this.labelManager = labelManager;
        this.pageManager = pageManager;
        this.eventPublisher = eventPublisher;
        this.resolver = resolver;
    }

    @Override
    public BlueprintModuleDescriptor getBlueprintDescriptor(final ModuleCompleteKey blueprintKey) {
        final BlueprintModuleDescriptor moduleDescriptor = (BlueprintModuleDescriptor) pluginAccessor.getEnabledPluginModule(blueprintKey
                .getCompleteKey());
        checkNotNull(moduleDescriptor, "module descriptor not found [key='" + blueprintKey + "']");
        return moduleDescriptor;
    }

    @Override
    @Deprecated
    public Page createAndPinIndexPage(final BlueprintDescriptor blueprintDescriptor, final Space space) {
        if (blueprintDescriptor.isIndexDisabled())
            return null;

        checkNotNull(space, "space must be non-null");
        checkNotNull(blueprintDescriptor, "blueprintDescriptor must be non-null");

        String indexPageTitle = getIndexPageTitle(blueprintDescriptor);
        assert (StringUtils.isNotBlank(indexPageTitle));

        final Page indexPage = indexPageManager.getOrCreateIndexPage(blueprintDescriptor, space, indexPageTitle);
        if (sidebarServiceAvailable()) {
            pinIndexPageToSidebar(blueprintDescriptor.getIndexKey(), space, indexPage);
        }

        return indexPage;
    }

    @Override
    public Page createAndPinIndexPage(final ContentBlueprint blueprint, final Space space) {
        if (blueprint.isIndexDisabled())
            return null;

        checkNotNull(space, "space must be non-null");
        checkNotNull(blueprint, "blueprint must be non-null");

        String indexPageTitle = getIndexPageTitle(blueprint);
        assert (StringUtils.isNotBlank(indexPageTitle));

        // CONFDEV-19787: If we don't synchronize, we can end up having multiple clones in the DB
        final UUID uuid = blueprint.getId();
        final BlueprintLock newLock = new BlueprintLock(uuid, space);
        Object lock = getOrCreateLocks.putIfAbsent(newLock, newLock);
        if (lock == null) {
            lock = newLock;
        }

        synchronized (lock) {
            Page indexPage = indexPageManager.findIndexPage(blueprint, space);
            if (indexPage == null) {
                indexPage = indexPageManager.createIndexPage(blueprint, space, indexPageTitle);
                if (sidebarServiceAvailable()) {
                    pinIndexPageToSidebar(blueprint.getIndexKey(), space, indexPage);
                }
            }

            return indexPage;
        }
    }

    private boolean sidebarServiceAvailable() {
        return sidebarLinkService != null;
    }

    private void pinIndexPageToSidebar(final String indexKey, final Space space, final Page indexPage) {
        try {
            if (!sidebarLinkToPageExists(space, indexPage)) {
                // We add two classes so that we'll have CSS/JS control over *all* index links.
                final String iconClass = "blueprint " + indexKey;
                sidebarLinkService.forceCreate(space.getKey(), indexPage.getId(), null, null, iconClass);
            }
        } catch (Exception e) {
            log.info("Error pinning page", e); // an error here shouldn't prevent content from being created
        }
    }

    private boolean sidebarLinkToPageExists(final Space space, final Page page) throws NotPermittedException {
        return sidebarLinkService.hasQuickLink(space.getKey(), page.getId());
    }

    @Override
    public String getIndexPageTitle(final BlueprintDescriptor blueprintDescriptor) {
        final String indexTitleI18nKey = getIndexPageTitleKey(blueprintDescriptor.getIndexTitleI18nKey(), blueprintDescriptor.getBlueprintKey().getCompleteKey());
        return i18nResolver.getText(indexTitleI18nKey);
    }

    @Override
    public String getIndexPageTitle(final ContentBlueprint blueprint) {
        final String indexTitleI18nKey = getIndexPageTitleKey(blueprint.getIndexTitleI18nKey(), blueprint.getModuleCompleteKey());
        return i18nResolver.getText(indexTitleI18nKey);
    }

    @Override
    public Page createBlueprintPage(ContentBlueprint blueprint, ConfluenceUser user, Space space, Page parentPage,
                                    Map<String, Object> context) {
        final Page indexPage = createAndPinIndexPage(blueprint, space);
        if (parentPage == null) {
            // If the Blueprint page is not created from another page, it will be added under the Index page.
            parentPage = indexPage;
        }

        ContentTemplateRef contentTemplateRef = getContentTemplateRef(blueprint, context);
        final Page blueprintPage = contentGenerator.generateBlueprintPageObject(contentTemplateRef, space, context);

        //TODO: Handle duplicate page titles gracefully. At the moment Confluence throws a com.atlassian.confluence.pages.DuplicateDataRuntimeException.
        // Override the default title if one was passed in the request
        String blueprintPageCustomTitle = (String) context.get(ContentBlueprintService.PAGE_TITLE);
        if (isNotBlank(blueprintPageCustomTitle)) {
            blueprintPage.setTitle(blueprintPageCustomTitle);
        }

        String indexKey = blueprint.getIndexKey();
        if (isNotBlank(indexKey)) {
            Label blueprintLabel = new Label(indexKey);
            labelManager.addLabel((Labelable) blueprintPage, blueprintLabel);
        }

        if (parentPage != null)
            parentPage.addChild(blueprintPage);

        pageManager.saveContentEntity(blueprintPage, DefaultSaveContext.DEFAULT);

        eventPublisher.publish(new BlueprintPageCreateEvent(this, blueprintPage, blueprint, user, context));

        // TODO - permissions and other things we might have in the context object.

        return blueprintPage;
    }

    @Override
    public Page createPageFromTemplate(ContentTemplateRef contentTemplateRef, ConfluenceUser user, Space space, Page parentPage,
                                       Map<String, Object> context) {
        return this.createPageFromTemplate(contentTemplateRef, user, space, parentPage, context, DefaultSaveContext.DEFAULT);
    }

    @Override
    public Page createPageFromTemplate(final ContentTemplateRef contentTemplateRef,
                                       final ConfluenceUser user,
                                       final Space space,
                                       final Page parentPage,
                                       final Map<String, Object> context,
                                       final SaveContext saveContext) {
        final Page result = contentGenerator.generateBlueprintPageObject(contentTemplateRef, space, context);

        if (parentPage != null) {
            parentPage.addChild(result);
        }
        pageManager.saveContentEntity(result, saveContext);
        eventPublisher.publish(new TemplatePageCreateEvent(this, result, contentTemplateRef, user, context, saveContext));

        return result;
    }

    private static ContentTemplateRef getContentTemplateRef(ContentBlueprint blueprint, Map<String, Object> context) {
        String contentTemplateOverride = (String) context.get(CONTENT_TEMPLATE_REF_ID_CONTEXT_KEY);
        if (StringUtils.isNotBlank(contentTemplateOverride)) {
            UUID contentTemplateRefId = UUID.fromString(contentTemplateOverride);
            return findContentTemplateRefInBlueprint(blueprint, contentTemplateRefId);
        }

        return blueprint.getFirstContentTemplateRef();
    }

    private static ContentTemplateRef findContentTemplateRefInBlueprint(ContentBlueprint contentBlueprint, UUID refId) {
        for (ContentTemplateRef ref : contentBlueprint.getContentTemplateRefs()) {
            if (ref.getId().equals(refId)) {
                return ref;
            }
        }
        throw new IllegalStateException("Content blueprint has no ContentTemplateRef with id: " + refId);
    }

    private String getIndexPageTitleKey(String indexTitleI18nKey, String blueprintModuleKey) {
        if (isBlank(indexTitleI18nKey)) {
            // TODO - FLAWED. Child blueprints might not have Web-items.
            // fairly likely - the plugin dev probably wants the same name for the index page as for the blueprintModuleDescriptor in the
            // dialog, so we use the web-item for this blueprintModuleDescriptor.
            final WebItemModuleDescriptor webItem = resolver.getWebItemMatchingBlueprint(blueprintModuleKey);
            if (webItem != null) {
                return webItem.getI18nNameKey();
            }
        }
        return indexTitleI18nKey;
    }

    @Override
    public String getBlueprintKeyForContent(final AbstractPage page) {
        return contentPropertyManager.getStringProperty(page, BLUEPRINT_KEY);
    }
}
