package com.atlassian.confluence.plugins.createcontent.actions;

import com.atlassian.confluence.pages.Page;
import com.atlassian.confluence.pages.templates.PageTemplate;
import com.atlassian.confluence.pages.templates.PluginTemplateReference;
import com.atlassian.confluence.pages.templates.TemplateHandler;
import com.atlassian.confluence.pages.templates.variables.StringVariable;
import com.atlassian.confluence.pages.templates.variables.Variable;
import com.atlassian.confluence.plugins.createcontent.api.contextproviders.AbstractBlueprintContextProvider;
import com.atlassian.confluence.plugins.createcontent.api.services.ContentBlueprintService;
import com.atlassian.confluence.plugins.createcontent.extensions.ContentTemplateModuleDescriptor;
import com.atlassian.confluence.plugins.createcontent.impl.ContentTemplateRef;
import com.atlassian.confluence.plugins.createcontent.services.model.CreateBlueprintPageRequest;
import com.atlassian.confluence.plugins.createcontent.template.PluginPageTemplateHelper;
import com.atlassian.confluence.spaces.Space;
import com.atlassian.plugin.ModuleCompleteKey;
import com.atlassian.plugin.ModuleDescriptor;
import com.atlassian.plugin.PluginAccessor;
import com.atlassian.plugin.spring.scanner.annotation.export.ExportAsService;
import com.atlassian.plugin.spring.scanner.annotation.imports.ComponentImport;
import com.atlassian.plugin.web.ContextProvider;
import com.atlassian.plugin.web.NoOpContextProvider;
import com.atlassian.sal.api.message.I18nResolver;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.io.StringReader;
import java.util.List;
import java.util.Map;

import static com.google.common.base.Preconditions.checkNotNull;
import static com.google.common.collect.Lists.newLinkedList;
import static org.apache.commons.lang3.StringUtils.EMPTY;
import static org.apache.commons.lang3.StringUtils.isBlank;
import static org.apache.commons.lang3.StringUtils.isNotBlank;

/**
 * Responsible for generating content for created blueprint pages.
 */
@Component
@ExportAsService(BlueprintContentGenerator.class)
public class DefaultBlueprintContentGenerator implements BlueprintContentGenerator {
    private static final Logger log = LoggerFactory.getLogger(DefaultBlueprintContentGenerator.class);

    public static final String CONTENT_PAGE_TITLE_CONTEXT_KEY = "ContentPageTitle";
    public static final String PAGE_TITLE_PREFIX_CONTEXT_KEY = "ParentPageTitle";
    public static final String CONTENT_TEMPLATE_REF_ID_CONTEXT_KEY = "contentTemplateRefId";

    public static final String USE_PAGE_TEMPLATE_TITLE_CONTEXT_KEY = "UsePageTemplateNameForTitle";

    private final PluginAccessor pluginAccessor;
    private final PluginPageTemplateHelper pluginPageTemplateHelper;
    private final TemplateHandler templateHandler;
    private final I18nResolver i18nResolver;

    @Autowired
    public DefaultBlueprintContentGenerator(
            final @ComponentImport PluginAccessor pluginAccessor,
            final PluginPageTemplateHelper pluginPageTemplateHelper,
            final @ComponentImport TemplateHandler templateHandler,
            final @ComponentImport I18nResolver i18nResolver) {
        this.pluginAccessor = pluginAccessor;
        this.pluginPageTemplateHelper = pluginPageTemplateHelper;
        this.templateHandler = templateHandler;
        this.i18nResolver = i18nResolver;
    }

    /**
     * @deprecated since 2.0.0
     */
    @Override
    @Deprecated
    public Page generateBlueprintPageObject(PluginTemplateReference pluginTemplateReference, Map<String, ?> context) {
        ModuleCompleteKey contentTemplateKey = pluginTemplateReference.getModuleCompleteKey();
        ContentTemplateModuleDescriptor contentTemplateModule = getContentTemplateModuleDescriptor(contentTemplateKey.getCompleteKey());
        Map<String, Object> combinedContext = getContentTemplateContext(contentTemplateModule, context);
        final PageTemplate contentTemplate = pluginPageTemplateHelper.getPageTemplate(pluginTemplateReference);

        final Page page = new Page();

        page.setTitle(getContentPageTitle(contentTemplate, combinedContext));
        page.setBodyAsString(renderTemplate(contentTemplate, combinedContext));
        page.setSpace(pluginTemplateReference.getSpace());

        return page;
    }

    @Override
    public Page generateBlueprintPageObject(CreateBlueprintPageRequest createRequest) {
        Page page = generateBlueprintPageObject(createRequest.getContentTemplateRef(), createRequest.getSpace(),
                createRequest.getContext());

        // TODO: Handle duplicate page titles gracefully. At the moment Confluence throws a com.atlassian.confluence.pages.DuplicateDataRuntimeException.
        // Override the default title if one was passed in the request
        String blueprintPageCustomTitle = createRequest.getTitle();
        if (isBlank(blueprintPageCustomTitle)) {
            blueprintPageCustomTitle = (String) createRequest.getContext().get(ContentBlueprintService.PAGE_TITLE);
        }
        if (isNotBlank(blueprintPageCustomTitle)) {
            page.setTitle(blueprintPageCustomTitle);
        }

        return page;
    }

    @Override
    public Page generateBlueprintPageObject(ContentTemplateRef contentTemplateRef, Space space, Map<String, Object> context) {
        Map<String, Object> combinedContext;
        final String moduleCompleteKey = contentTemplateRef.getModuleCompleteKey();
        if (StringUtils.isNotBlank(moduleCompleteKey)) {
            ContentTemplateModuleDescriptor contentTemplateModule = getContentTemplateModuleDescriptor(moduleCompleteKey);
            combinedContext = getContentTemplateContext(contentTemplateModule, context);
        } else {
            combinedContext = context;
        }
        final PageTemplate contentTemplate = pluginPageTemplateHelper.getPageTemplate(contentTemplateRef);

        final Page page = new Page();

        page.setTitle(getContentPageTitle(contentTemplate, combinedContext));
        page.setBodyAsString(renderTemplate(contentTemplate, combinedContext));
        page.setSpace(space);

        return page;
    }

    private String getContentPageTitle(PageTemplate contentTemplate, Map<String, Object> combinedContext) {
        String title = (String) combinedContext.get(CONTENT_PAGE_TITLE_CONTEXT_KEY);
        if (isBlank(title)) {
            // TODO: Why do we need to check for some random ContextKey, instead of always doing this when title is already empty? - Edu
            Object useTemplateNameObj = combinedContext.get(USE_PAGE_TEMPLATE_TITLE_CONTEXT_KEY);
            Boolean useTemplateName = useTemplateNameObj instanceof Boolean ? (Boolean) useTemplateNameObj : Boolean.parseBoolean((String) useTemplateNameObj);
            if (useTemplateName != null && useTemplateName) {
                title = i18nResolver.getText(contentTemplate.getTitle());
            } else {
                return EMPTY;
            }
        }

        final String titlePrefix = (String) combinedContext.get(PAGE_TITLE_PREFIX_CONTEXT_KEY);
        if (isNotBlank(titlePrefix)) {
            return titlePrefix + " - " + title;
        }

        return title;
    }

    /**
     * Renders the plugin PageTemplate identified by the moduleCompleteKey and space.
     * <p>
     * "Render" here means taking the storage format of the template, filling in any template variables with values
     * from the specified context (as well as any <tt>ContextProvider</tt>s).
     * <p>
     * Plugin PageTemplate's can be modified and their modification is saved in the database against a particular
     * Confluence space. This is where the specified becomes relevant, as it allows us to query and find a these modified templates.
     *
     * @param contentTemplate the key identifying the plugin PageTemplate
     * @param context         additional context
     * @return the storage format of the template with variables inserted
     */
    private String renderTemplate(PageTemplate contentTemplate, Map<String, Object> context) {
        final List<Variable> variables = buildTemplateVariables(context);
        final String templateContent = contentTemplate.getContent();

        return templateHandler.insertVariables(new StringReader(templateContent), variables);
    }

    private static List<Variable> buildTemplateVariables(Map<String, Object> contextMap) {
        final List<Variable> variables = newLinkedList();

        for (Map.Entry<String, Object> contextEntry : contextMap.entrySet()) {
            final String value = contextEntry.getValue() != null ? contextEntry.getValue().toString() : EMPTY;
            variables.add(new StringVariable(contextEntry.getKey(), value));
        }
        return variables;
    }

    private Map<String, Object> getContentTemplateContext(ContentTemplateModuleDescriptor moduleDescriptor, Map<String, ?> context) {
        // contextProvider will never be null - will at least be a NoOpContextProvider that returns the context it is
        // passed.
        ContextProvider contextProvider = moduleDescriptor.getContextProvider();
        if (!(contextProvider instanceof AbstractBlueprintContextProvider ||
                contextProvider instanceof NoOpContextProvider)) {
            log.warn("This Blueprint ContextProvider class should extend AbstractBlueprintContextProvider: " + contextProvider.getClass().getName());
        }

        // Don't replace the incoming context with a new map - the context passed to generateBlueprintPageObject
        // needs to be updated so that events including the full context can be published.
        // Yes, this does mean that we're relying on a side effect, and we should really have a method on this
        // interface that returns a combined context to the same level that fires the create event.
        // However, this class is very likely to be refactored as part of the upcoming Space BP work so we might
        // hold off for now.
        return contextProvider.getContextMap((Map<String, Object>) context);
    }

    private ContentTemplateModuleDescriptor getContentTemplateModuleDescriptor(String contentTemplateKey) {
        ModuleDescriptor moduleDescriptor = pluginAccessor.getEnabledPluginModule(contentTemplateKey);
        checkNotNull(moduleDescriptor, "module descriptor not found [key='" + contentTemplateKey + "']");
        return (ContentTemplateModuleDescriptor) moduleDescriptor;
    }

    @Override
    public Page createIndexPageObject(PluginTemplateReference pluginTemplateReference, Map<String, Object> context) {
        final PageTemplate contentTemplate = pluginPageTemplateHelper.getPageTemplate(pluginTemplateReference);
        String moduleCompleteKey = pluginTemplateReference.getModuleCompleteKey().getCompleteKey();

        return createIndexPageObject(pluginTemplateReference.getSpace(), context, contentTemplate, moduleCompleteKey);
    }

    @Override
    public Page createIndexPageObject(ContentTemplateRef contentTemplateRef, Space space, Map<String, Object> context) {
        final PageTemplate contentTemplate = pluginPageTemplateHelper.getPageTemplate(contentTemplateRef);
        String moduleCompleteKey = contentTemplateRef.getModuleCompleteKey();

        return createIndexPageObject(space, context, contentTemplate, moduleCompleteKey);
    }

    private Page createIndexPageObject(Space space, Map<String, Object> context, PageTemplate contentTemplate, final String moduleCompleteKey) {
        ContentTemplateModuleDescriptor contentTemplateModule = getContentTemplateModuleDescriptor(moduleCompleteKey);
        Map<String, Object> combinedContext = getContentTemplateContext(contentTemplateModule, context);

        final String indexPageContent = renderTemplate(contentTemplate, combinedContext);

        final Page indexPage = new Page();
        indexPage.setBodyAsString(indexPageContent);
        indexPage.setSpace(space);

        Page spaceHomePage = space.getHomePage();
        if (spaceHomePage != null) {
            spaceHomePage.addChild(indexPage);
        }

        return indexPage;
    }
}
