package com.atlassian.confluence.plugins.createcontent.rest;

import com.atlassian.confluence.api.service.accessmode.AccessModeService;
import com.atlassian.confluence.api.service.exceptions.ReadOnlyException;
import com.atlassian.confluence.plugins.createcontent.api.exceptions.ResourceErrorType;
import com.atlassian.confluence.plugins.createcontent.exceptions.ResourceException;
import com.atlassian.confluence.security.Permission;
import com.atlassian.confluence.security.PermissionManager;
import com.atlassian.confluence.spaces.Space;
import com.atlassian.confluence.spaces.SpaceManager;
import com.atlassian.confluence.user.AuthenticatedUserThreadLocal;
import com.atlassian.confluence.user.ConfluenceUser;
import com.atlassian.plugin.spring.scanner.annotation.imports.ComponentImport;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.ws.rs.Produces;
import javax.ws.rs.core.Response;

import static javax.ws.rs.core.MediaType.APPLICATION_JSON;
import static org.apache.commons.lang3.StringUtils.isBlank;

@Produces({APPLICATION_JSON})
public abstract class AbstractRestResource {
    private final SpaceManager spaceManager;
    protected final PermissionManager permissionManager;
    protected final AccessModeService accessModeService;

    protected AbstractRestResource(
            final @ComponentImport PermissionManager permissionManager,
            final @ComponentImport SpaceManager spaceManager,
            final @ComponentImport AccessModeService accessModeService) {
        this.spaceManager = spaceManager;
        this.permissionManager = permissionManager;
        this.accessModeService = accessModeService;
    }

    protected Space checkSpaceAdminPermission(@Nonnull String spaceKey) {
        if (accessModeService.isReadOnlyAccessModeEnabled()) {
            throw new ReadOnlyException();
        }
        final Space space = getAndCheckSpace(spaceKey);
        if (!permissionManager.hasPermission(getUser(), Permission.ADMINISTER, space)) {
            throw new ResourceException("Only space administrators for " + spaceKey + " can make this request.", Response.Status.BAD_REQUEST, ResourceErrorType.PERMISSION_USER_ADMIN_SPACE, spaceKey);
        }

        return space;
    }

    protected void checkAdminPermission() {
        if (!permissionManager.hasPermission(getUser(), Permission.ADMINISTER, PermissionManager.TARGET_APPLICATION)) {
            throw new ResourceException("Only site administrators can make this request.", Response.Status.BAD_REQUEST, ResourceErrorType.PERMISSION_USER_ADMIN);
        }
    }

    protected void checkEmptyParameter(final String parameter, final String parameterName) {
        if (isBlank(parameter)) {
            throwMissingParameterError(parameterName);
        }
    }

    protected void checkNullParameter(final Object parameter, final String parameterName) {
        if (parameter == null) {
            throwMissingParameterError(parameterName);
        }
    }

    protected void checkNullEntity(final Object entity) {
        if (entity == null) {
            // TODO: Maybe something better than "bodyEntity"? A new ResourceErrorType?
            throw new ResourceException("This request requires a body entity with the data of the object to create/update", Response.Status.BAD_REQUEST, ResourceErrorType.PARAMETER_MISSING, "bodyEntity");
        }
    }

    private void throwMissingParameterError(final String parameterName) {
        throw new ResourceException(String.format("Missing '%1$s' parameter", parameterName), Response.Status.BAD_REQUEST, ResourceErrorType.PARAMETER_MISSING, parameterName);
    }

    @Nonnull
    protected Space getAndCheckSpace(final String spaceKey) {
        final Space space = spaceManager.getSpace(spaceKey);

        if (space == null) {
            throw new ResourceException(String.format("No space found for spaceKey '%1$s'", spaceKey), Response.Status.BAD_REQUEST, ResourceErrorType.NOT_FOUND_SPACE, spaceKey);
        }

        return space;
    }

    @Nullable
    protected ConfluenceUser getUser() {
        return AuthenticatedUserThreadLocal.get();
    }
}
