package com.atlassian.confluence.plugins.createcontent.actions;

import com.atlassian.confluence.core.ContentEntityObject;
import com.atlassian.confluence.pages.DraftsTransitionHelper;
import com.atlassian.confluence.pages.Page;
import com.atlassian.confluence.plugins.createcontent.api.exceptions.BlueprintIllegalArgumentException;
import com.atlassian.confluence.plugins.createcontent.services.RequestResolver;
import com.atlassian.confluence.plugins.createcontent.services.RequestStorage;
import com.atlassian.confluence.plugins.createcontent.services.model.CreateBlueprintPageEntity;
import com.atlassian.confluence.plugins.createcontent.services.model.CreateBlueprintPageRequest;
import com.atlassian.spring.container.ContainerManager;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Loads the page editor for a given Blueprint scenario; then when the page is saved,
 * creates a index page and page pin.
 * <p>
 * Plugin developers wishing to create a new Blueprint with a minimum of code and standard
 * behaviour can use this action to skip some boilerplate.
 */
public final class CreateAndEditAction extends AbstractCreateBlueprintPageAction {
    private static final Logger log = LoggerFactory.getLogger(CreateAndEditAction.class);

    private RequestStorage requestStorage;
    private CreateBlueprintPageRequest createBlueprintPageRequest;
    private CreateBlueprintPageEntity blueprintPageEntity;
    private RequestResolver resolver;

    /**
     * Called when editing a Blueprint Draft that has already been created.
     */
    @Override
    public void setDraftId(long draftId) {
        super.setDraftId(draftId);

        // Resuming editing of a previously-saved Draft. We must set the newSpaceKey so that the inherited
        // permission checks work correctly. We need this h_ack in here because we have chosen not to pollute the
        // URL with the spaceKey when it can be surmised from the draft. dT
        setUseDraft(true); // TODO - this might hide the "resume draft" message if two blueprints of same type and title(?)
        // triggered too close together.

        // getDraftAsCEO requires the draftsTransitionHelper to be set but it
        // cannot be guaranteed to be injected at the time setDraftId is called
        if (draftsTransitionHelper == null) {
            setDraftsTransitionHelper((DraftsTransitionHelper) ContainerManager.getComponent("draftsTransitionHelper"));
        }
        ContentEntityObject draft = getDraftAsCEO();

        if (draft == null) {
            log.warn("Requested draft with id >{}< was not found", draftId);
            return;
        }

        setSpaceKey(DraftsTransitionHelper.getSpaceKey(draft));
        setNewSpaceKey(DraftsTransitionHelper.getSpaceKey(draft));
        setLabelsString(draft.getLabels());

        // Interestingly, the parentPageId is NOT STORED against the Draft object. This appears to be an existing bug
        // for non-Blueprint page drafts too, but it's exacerbated when we create and THEN edit the Draft as part of
        // standard Blueprint page creation.
        blueprintPageEntity = requestStorage.retrieveRequest(draft);
        if (blueprintPageEntity == null)
            return;

        long entityParentPageId = blueprintPageEntity.getParentPageId();
        setParentPageId(entityParentPageId);
        setFromPageId(entityParentPageId);
    }

    @Override
    public String doDefault() throws Exception {
        if (getDraftAsCEO() == null)
            return "pagenotfound";

        return super.doDefault();
    }

    /**
     * Creates a draft Blueprint page within the action and renders the editor.
     *
     * @deprecated since 2.1.0 Use the REST endpoint createDraft instead. Since v2.0.
     */
    @Deprecated
    public String doEdit() throws Exception {
        // Populates the action fields from the blueprint, suitable for going to the editor
        populateBlueprintPage();

        // Must call super.doDefault() else draft won't be initialised
        return super.doDefault();
    }

    @Override
    public void validate() {
        if (contentBlueprint == null) {
            // May be null if editing a Draft; get the blueprint from the stored create-request.
            contentBlueprint = getCreateBlueprintPageRequest().getContentBlueprint();
        }
        super.validate();
        if (!hasActionErrors()) {
            validatePageTitleAgainstIndexPageTitle();
        }
    }

    @Override
    protected String beforeAdd() throws Exception {
        Page indexPage = getOrCreateIndexPage();

        // If the Blueprint page is not created from another page and the index page exists, it will be added under the Index page for that
        // Blueprint.
        if (getParentPage() == null && indexPage != null) {
            setParentPageId(indexPage.getId());
        }
        return super.beforeAdd();
    }

    /**
     * Called after the user has saved the page in the editor
     */
    @Override
    protected String afterAdd() {
        final String result = super.afterAdd();

        if (!SUCCESS.equals(result)) {
            return result;
        } else {
            final Page page = (Page) getPage();
            // TODO - this 'if' should pass every time when we remove the old action-logic. dT
            if (context == null || context.isEmpty()) {
                context = getCreateBlueprintPageRequest().getContext();
            }
            ContentEntityObject draft = getDraftAsCEO();
            requestStorage.clear(draft);
            sendBlueprintPageCreateEvent(page);

            return SUCCESS;
        }
    }

    private CreateBlueprintPageRequest getCreateBlueprintPageRequest() {
        if (createBlueprintPageRequest == null) {
            if (blueprintPageEntity == null) {
                blueprintPageEntity = requestStorage.retrieveRequest(getDraftAsCEO());
            }
            try {
                createBlueprintPageRequest = resolver.resolve(blueprintPageEntity, getAuthenticatedUser());
            } catch (BlueprintIllegalArgumentException e) {
                // Not expected - indicated valid state of draft changed between creation and page save
                throw new RuntimeException(e);
            }
        }
        return createBlueprintPageRequest;
    }

    public void setRequestStorage(RequestStorage requestStorage) {
        this.requestStorage = requestStorage;
    }

    public void setRequestResolver(RequestResolver resolver) {
        this.resolver = resolver;
    }
}
