package com.atlassian.confluence.plugins.createcontent.upgrade;

import com.atlassian.confluence.pages.templates.PageTemplate;
import com.atlassian.confluence.plugins.createcontent.BlueprintConstants;
import com.atlassian.confluence.plugins.createcontent.BlueprintStateController;
import com.atlassian.confluence.plugins.createcontent.ContentBlueprintManager;
import com.atlassian.confluence.plugins.createcontent.impl.ContentBlueprint;
import com.atlassian.confluence.plugins.createcontent.services.TemplateUpdater;
import com.atlassian.confluence.plugins.createcontent.template.BlueprintsDisabledPredicate;
import com.atlassian.confluence.plugins.createcontent.template.PluginPageTemplateHelper;
import com.atlassian.plugin.ModuleCompleteKey;
import com.atlassian.plugin.ModuleDescriptor;
import com.atlassian.plugin.PluginAccessor;
import com.atlassian.plugin.PluginController;
import com.atlassian.plugin.spring.scanner.annotation.export.ExportAsService;
import com.atlassian.plugin.spring.scanner.annotation.imports.ComponentImport;
import com.atlassian.plugin.web.descriptors.WebItemModuleDescriptor;
import com.atlassian.sal.api.message.Message;
import com.atlassian.sal.api.upgrade.PluginUpgradeTask;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.UUID;

import static com.atlassian.confluence.plugins.createcontent.api.contextproviders.BlueprintContextKeys.BLUEPRINT_MODULE_KEY;

/**
 * Persists global-level user-customised blueprints state after upgrade.
 * - Globally disabled blueprints are persisted in Bandana an enabled in the plugin system.
 * - Globally user-customised blueprints.
 * <p>
 * Note: The reason of combining the above tasks is...
 * When we are upgrading Global Blueprints and sorting through the disabled ones, we are creating clones for them
 * and adding that to Bandana. However, if the global blueprints have been edited, we want to store the UUID of
 * the customised global blueprint and not the clone UUID.
 * Before combining the upgrade tasks, only template edited upgrade task knew about which templates were edited,
 * and creates the customised global blueprint. At this point, if the blueprint is disabled, the Bandana entry will
 * need to be modified.
 * Due of this coupling of customisation and disabling of global bps, we decided to combine the upgrade tasks.
 */
@Component
@ExportAsService(PluginUpgradeTask.class)
public class GlobalBlueprintsUpgradeTask implements PluginUpgradeTask {
    private final PluginAccessor pluginAccessor;

    private final ContentBlueprintManager contentBlueprintManager;

    private final PluginController pluginController;

    private final TemplateUpdater templateUpdater;

    private final BlueprintStateController blueprintStateController;

    private final PluginPageTemplateHelper pluginPageTemplateHelper;

    @Autowired
    public GlobalBlueprintsUpgradeTask(
            final @ComponentImport PluginAccessor pluginAccessor,
            final ContentBlueprintManager contentBlueprintManager,
            final @ComponentImport PluginController pluginController,
            final TemplateUpdater templateUpdater,
            final BlueprintStateController blueprintStateController,
            final PluginPageTemplateHelper pluginPageTemplateHelper) {
        this.pluginAccessor = pluginAccessor;
        this.contentBlueprintManager = contentBlueprintManager;
        this.pluginController = pluginController;
        this.templateUpdater = templateUpdater;
        this.blueprintStateController = blueprintStateController;
        this.pluginPageTemplateHelper = pluginPageTemplateHelper;
    }

    /**
     * The build number for this upgrade task. Once this upgrade task has run the plugin manager will store this
     * build number against this plugin type.  After this only upgrade tasks with higher build numbers will be run
     */
    @Override
    public int getBuildNumber() {
        return 2;
    }

    @Override
    public String getShortDescription() {
        return "Persists in Bandana all global disabled blueprints, enabled them back in the plugin system.";
    }

    @Override
    public Collection<Message> doUpgrade() {
        // Gets disabled web-item blueprints from the plugin system
        // noinspection unchecked
        Collection<WebItemModuleDescriptor> globallyDisabledBlueprintWebItems =
                pluginAccessor.getModuleDescriptors(new BlueprintsDisabledPredicate(pluginAccessor));

        List<ModuleCompleteKey> globallyDisabledBlueprintModuleKeys = Lists.newArrayList();

        for (WebItemModuleDescriptor webItemModuleDescriptor : globallyDisabledBlueprintWebItems) {
            // Enable blueprint web-item in the plugin system
            pluginController.enablePluginModule(webItemModuleDescriptor.getCompleteKey());

            final String blueprintModuleKey = getBlueprintModuleKey(webItemModuleDescriptor);
            final ModuleCompleteKey blueprintModuleCompleteKey = new ModuleCompleteKey(
                    webItemModuleDescriptor.getPluginKey(), blueprintModuleKey);

            globallyDisabledBlueprintModuleKeys.add(blueprintModuleCompleteKey);

            // Gets content blueprint if existent, otherwise it creates an AO clone.
            contentBlueprintManager.getPluginBlueprint(blueprintModuleCompleteKey);
        }

        updateGlobalTemplates();

        saveGlobalDisabledBlueprintState(globallyDisabledBlueprintModuleKeys);

        return null;
    }

    private void saveGlobalDisabledBlueprintState(List<ModuleCompleteKey> globallyDisabledBlueprintModuleKeys) {
        Set<UUID> globallyDisabledBlueprintIds = Sets.newHashSet();

        for (ModuleCompleteKey disabledBlueprintModuleKey : globallyDisabledBlueprintModuleKeys) {
            ContentBlueprint disabledBlueprint = contentBlueprintManager.getPluginBackedContentBlueprint(
                    disabledBlueprintModuleKey, null);

            if (disabledBlueprint != null)
                globallyDisabledBlueprintIds.add(disabledBlueprint.getId());
        }

        blueprintStateController.disableBlueprints(globallyDisabledBlueprintIds, null);
    }


    private void updateGlobalTemplates() {
        // Get all global-level templates declared inside a blueprint
        List<PageTemplate> globallyCustomisedTemplates = pluginPageTemplateHelper.getPageTemplates(null);
        updateBlueprintContentTemplates(globallyCustomisedTemplates);
    }

    private String getBlueprintModuleKey(ModuleDescriptor webItemModuleDescriptor) {
        // noinspection unchecked
        final Map<String, String> moduleDescriptorParams = webItemModuleDescriptor.getParams();
        return moduleDescriptorParams.get(BLUEPRINT_MODULE_KEY.key());
    }

    // Upgrades the content template referenced if it belongs to a blueprint.
    private void updateBlueprintContentTemplates(List<PageTemplate> pageTemplates) {
        for (PageTemplate pageTemplate : pageTemplates) {
            templateUpdater.updateContentTemplateRef(pageTemplate);
        }
    }

    @Override
    public String getPluginKey() {
        return BlueprintConstants.PLUGIN_KEY;
    }

}
