package com.atlassian.confluence.plugins.createcontent.rest;

import com.atlassian.confluence.api.service.accessmode.AccessModeService;
import com.atlassian.confluence.api.service.exceptions.ReadOnlyException;
import com.atlassian.confluence.core.BodyType;
import com.atlassian.confluence.pages.templates.PageTemplate;
import com.atlassian.confluence.pages.templates.PageTemplateManager;
import com.atlassian.confluence.plugins.createcontent.ContentTemplateRefManager;
import com.atlassian.confluence.plugins.createcontent.api.exceptions.ResourceErrorType;
import com.atlassian.confluence.plugins.createcontent.exceptions.ResourceException;
import com.atlassian.confluence.plugins.createcontent.impl.ContentTemplateRef;
import com.atlassian.confluence.plugins.createcontent.template.PluginPageTemplateHelper;
import com.atlassian.confluence.security.Permission;
import com.atlassian.confluence.security.PermissionManager;
import com.atlassian.confluence.spaces.Space;
import com.atlassian.confluence.spaces.SpaceManager;
import com.atlassian.confluence.user.AuthenticatedUserThreadLocal;
import com.atlassian.plugin.spring.scanner.annotation.imports.ComponentImport;
import org.apache.commons.lang3.StringUtils;

import javax.annotation.Nonnull;
import javax.ws.rs.Consumes;
import javax.ws.rs.DELETE;
import javax.ws.rs.GET;
import javax.ws.rs.POST;
import javax.ws.rs.PUT;
import javax.ws.rs.Path;
import javax.ws.rs.QueryParam;
import javax.ws.rs.core.Response;
import java.util.UUID;

import static javax.ws.rs.core.MediaType.APPLICATION_JSON;
import static javax.ws.rs.core.MediaType.APPLICATION_XML;

@Path("/templates")
public class PageTemplateResource extends AbstractRestResource {
    public static final String PARAM_ID = "id";
    public static final String PARAM_CONTENT_TEMPLATE_REF_ID = "contentTemplateRefId";

    private final PageTemplateManager pageTemplateManager;
    private final PluginPageTemplateHelper pageTemplateHelper;
    private final ContentTemplateRefManager contentTemplateRefManager;
    private final SpaceManager spaceManager;

    public PageTemplateResource(
            final @ComponentImport PageTemplateManager pageTemplateManager,
            final PluginPageTemplateHelper pageTemplateHelper,
            final ContentTemplateRefManager contentTemplateRefManager,
            final @ComponentImport SpaceManager spaceManager,
            final @ComponentImport PermissionManager permissionManager,
            final @ComponentImport AccessModeService accessModeService) {
        super(permissionManager, spaceManager, accessModeService);

        this.pageTemplateManager = pageTemplateManager;
        this.pageTemplateHelper = pageTemplateHelper;
        this.contentTemplateRefManager = contentTemplateRefManager;
        this.spaceManager = spaceManager;
    }

    @GET
    public PageTemplateForm get(@QueryParam(PARAM_ID) Integer id, @QueryParam(PARAM_CONTENT_TEMPLATE_REF_ID) UUID contentTemplateRefId) {
        checkIds(id, contentTemplateRefId);
        PageTemplate pageTemplate = null;
        if (id != null) {
            pageTemplate = getPageTemplate(id);
        } else if (contentTemplateRefId != null) {
            pageTemplate = getPageTemplate(contentTemplateRefId);
        }

        if (pageTemplate != null) {
            checkViewPermission(pageTemplate);
            return build(pageTemplate);
        }

        throw new ResourceException("Either 'id' or 'contentTemplateRefId' parameter should be specified", Response.Status.BAD_REQUEST, ResourceErrorType.PARAMETER_MISSING);
    }

    @POST
    @Consumes({APPLICATION_JSON, APPLICATION_XML})
    public long create(PageTemplateForm pageTemplate) {
        checkNullEntity(pageTemplate);

        PageTemplate newPageTemplate = build(pageTemplate);
        checkCreatePermission(newPageTemplate);
        pageTemplateManager.savePageTemplate(newPageTemplate, null);

        return newPageTemplate.getId();
    }

    private PageTemplate build(PageTemplateForm pageTemplate) {
        // FIXME: The Form object is not complete, we're missing labels and what-not
        PageTemplate result = new PageTemplate();

        result.setId(pageTemplate.id);
        result.setName(pageTemplate.name);
        result.setDescription(pageTemplate.description);
        result.setBodyType(BodyType.XHTML); // For now, all our objects are XHTML
        if (!StringUtils.isEmpty(pageTemplate.spaceKey)) {
            Space space = spaceManager.getSpace(pageTemplate.spaceKey);
            result.setSpace(space);
        }
        result.setVersion(pageTemplate.version);
        result.setContent(pageTemplate.content);

        return result;
    }

    private PageTemplateForm build(PageTemplate pageTemplate) {
        // FIXME: The Form object is not complete, we're missing labels and what-not
        PageTemplateForm result = new PageTemplateForm();

        result.id = pageTemplate.getId();
        result.name = pageTemplate.getName();
        result.description = pageTemplate.getDescription();
        Space space = pageTemplate.getSpace();
        if (space != null) {
            result.spaceKey = space.getKey();
        }
        result.version = pageTemplate.getVersion();
        result.content = pageTemplate.getContent();

        return result;
    }

    private void build(PageTemplateForm pageTemplate, PageTemplate originalPageTemplate) {
        if (!StringUtils.isEmpty(pageTemplate.name)) {
            originalPageTemplate.setName(pageTemplate.name);
        }
        if (!StringUtils.isEmpty(pageTemplate.description)) {
            originalPageTemplate.setDescription(pageTemplate.description);
        }
        if (!StringUtils.isEmpty(pageTemplate.content)) {
            originalPageTemplate.setContent(pageTemplate.content);
        }
        // Ignoring spaceKey, as you're not supposed to change spaces between PageTemplate versions
    }

    @PUT
    @Consumes({APPLICATION_JSON, APPLICATION_XML})
    public long update(PageTemplateForm pageTemplate) throws CloneNotSupportedException {
        PageTemplate originalPageTemplate = pageTemplateManager.getPageTemplate(pageTemplate.id);
        if (originalPageTemplate == null) {
            throw new ResourceException("Page template doesn't exist already", Response.Status.NOT_FOUND, ResourceErrorType.NOT_FOUND_PAGE_TEMPLATE, pageTemplate.id);
        }
        checkUpdatePermission(originalPageTemplate);
        PageTemplate newPageTemplate = (PageTemplate) originalPageTemplate.clone();
        build(pageTemplate, originalPageTemplate);
        pageTemplateManager.savePageTemplate(originalPageTemplate, newPageTemplate);

        return newPageTemplate.getId();
    }

    @DELETE
    public void delete(@QueryParam("id") Integer id, @QueryParam("contentTemplateRefId") UUID contentTemplateRefId) {
        checkIds(id, contentTemplateRefId);

        PageTemplate pageTemplate = null;

        if (id != null)
            pageTemplate = getPageTemplate(id);
        if (contentTemplateRefId != null)
            pageTemplate = getPageTemplate(contentTemplateRefId);

        if (pageTemplate == null)
            throw new ResourceException("The specified page template was not found", Response.Status.NOT_FOUND, ResourceErrorType.NOT_FOUND_PAGE_TEMPLATE);

        checkDeletePermission(pageTemplate);
        pageTemplateManager.removePageTemplate(pageTemplate);
    }

    private void checkIds(Integer id, UUID contentTemplateRefId) {
        if (id == null && contentTemplateRefId == null)
            throw new ResourceException("Either 'id' or 'contentTemplateRefId' parameter should be specified", Response.Status.BAD_REQUEST, ResourceErrorType.PARAMETER_MISSING);
        if (id != null && contentTemplateRefId != null)
            throw new ResourceException("Only one of 'id' or 'contentTemplateRefId' parameters should be specified", Response.Status.BAD_REQUEST, ResourceErrorType.PARAMETER_TOO_MANY);
    }

    private PageTemplate getPageTemplate(UUID contentTemplateRefId) {
        ContentTemplateRef contentTemplateRef = contentTemplateRefManager.getById(contentTemplateRefId);
        if (contentTemplateRef == null) {
            throw new ResourceException("The specified ContentTemplateRef was not found", Response.Status.NOT_FOUND, ResourceErrorType.NOT_FOUND_CONTENT_TEMPLATE_REF, contentTemplateRefId);
        }
        return pageTemplateHelper.getPageTemplate(contentTemplateRef);
    }

    private PageTemplate getPageTemplate(Integer id) {
        return pageTemplateManager.getPageTemplate(id);
    }

    private void checkDeletePermission(@Nonnull  PageTemplate pageTemplate) {
        checkPermission(pageTemplate, Permission.ADMINISTER, "You are not permitted to delete page template ");
    }

    private void checkUpdatePermission(@Nonnull  PageTemplate pageTemplate) {
        checkPermission(pageTemplate, Permission.ADMINISTER, "You are not permitted to update page template ");
    }

    private void checkCreatePermission(@Nonnull  PageTemplate pageTemplate) {
        checkPermission(pageTemplate, Permission.ADMINISTER, "You are not permitted to create page template ");
    }

    private void checkViewPermission(@Nonnull PageTemplate pageTemplate) {
        checkPermission(pageTemplate, Permission.VIEW, "You are not permitted to view page template ");
    }

    private void checkPermission(@Nonnull PageTemplate pageTemplate, Permission permission, String errorMessage) {
        if (accessModeService.isReadOnlyAccessModeEnabled() && !permission.equals(Permission.VIEW)) {
            throw new ReadOnlyException();
        }

        Space space = pageTemplate.getSpace();
        Object target = (space != null) ? space : PermissionManager.TARGET_APPLICATION;

        if (!permissionManager.hasPermission(AuthenticatedUserThreadLocal.get(), permission, target)) {
            throw new ResourceException(errorMessage + pageTemplate.getId(), Response.Status.FORBIDDEN,
                    AuthenticatedUserThreadLocal.get() == null ? ResourceErrorType.PERMISSION_ANONYMOUS_CREATE : ResourceErrorType.PERMISSION_USER_CREATE, space != null ? space.getKey(): null);
        }
    }
}
