package com.atlassian.confluence.plugins.createcontent.model;

import com.google.common.base.Objects;

/**
 * Represents the state of a blueprint in a given context.
 *
 * @since 2.1.16
 */
public class BlueprintState {
    public static final BlueprintState FULLY_ENABLED = new BlueprintState(false, false, false, false);
    public static final BlueprintState FULLY_DISABLED = new BlueprintState(true, true, true, true);

    private final boolean disabledGlobally;
    private final boolean disabledInSpace;
    private final boolean disabledInPluginSystem;
    private final boolean disabledByWebInterfaceManager;

    private BlueprintState(final boolean disabledGlobally, final boolean disabledInSpace,
                           final boolean disabledInPluginSystem, final boolean disabledByWebInterfaceManager) {
        this.disabledGlobally = disabledGlobally;
        this.disabledInSpace = disabledInSpace;
        this.disabledInPluginSystem = disabledInPluginSystem;
        this.disabledByWebInterfaceManager = disabledByWebInterfaceManager;
    }

    /**
     * @return true if the blueprint was disabled in the global templates screen, false otherwise.
     */
    public boolean isDisabledGlobally() {
        return disabledGlobally;
    }

    /**
     * @return true if the blueprint was disabled in the space templates admin screen, false otherwise.
     */
    public boolean isDisabledInSpace() {
        return disabledInSpace;
    }

    /**
     * @return true if the plugin backing the blueprint is not enabled, false otherwise.
     */
    public boolean isDisabledInPluginSystem() {
        return disabledInPluginSystem;
    }

    /**
     * @return true if the web item linking to the blueprint is returned by the web interface manager for the create
     * dialog section, false otherwise.
     */
    public boolean isDisabledByWebInterfaceManager() {
        return disabledByWebInterfaceManager;
    }

    /**
     * @param blueprintStates the blueprintStates to compare against.
     * @return true if at least one of the blueprint states match this one, false otherwise.
     */
    public boolean in(BlueprintState... blueprintStates) {
        for (BlueprintState blueprintState : blueprintStates) {
            if (this.equals(blueprintState))
                return true;
        }

        return false;
    }

    @Override
    public int hashCode() {
        return Objects.hashCode(disabledGlobally, disabledInSpace, disabledInPluginSystem,
                disabledByWebInterfaceManager);
    }

    @Override
    public boolean equals(Object obj) {
        if (obj == this)
            return true;

        if (!(obj instanceof BlueprintState))
            return false;

        BlueprintState that = (BlueprintState) obj;

        return (this.disabledGlobally == that.disabledGlobally) &&
                (this.disabledInSpace == that.disabledInSpace) &&
                (this.disabledInPluginSystem == that.disabledInPluginSystem) &&
                (this.disabledByWebInterfaceManager == that.disabledByWebInterfaceManager);
    }

    /**
     * A class that helps build {@link BlueprintState} objects.
     */
    public static class Builder {
        private boolean disabledGlobally;
        private boolean disabledInSpace;
        private boolean disabledInPluginSystem;
        private boolean disabledByWebInterfaceManager;

        public Builder disabledGlobally(boolean disabledGlobally) {
            this.disabledGlobally = disabledGlobally;
            return this;
        }

        public Builder disabledInSpace(boolean disabledInSpace) {
            this.disabledInSpace = disabledInSpace;
            return this;
        }

        public Builder disabledInPluginSystem(boolean disabledInPluginSystem) {
            this.disabledInPluginSystem = disabledInPluginSystem;
            return this;
        }

        public Builder disabledByWebInterfaceManager(boolean disabledByWebInterfaceManager) {
            this.disabledByWebInterfaceManager = disabledByWebInterfaceManager;
            return this;
        }

        public BlueprintState build() {
            return new BlueprintState(disabledGlobally, disabledInSpace, disabledInPluginSystem,
                    disabledByWebInterfaceManager);
        }
    }
}
