package com.atlassian.confluence.plugins.createcontent.listeners;

import com.atlassian.confluence.event.events.template.TemplateRemoveEvent;
import com.atlassian.confluence.event.events.template.TemplateUpdateEvent;
import com.atlassian.confluence.pages.templates.PageTemplate;
import com.atlassian.confluence.plugins.createcontent.api.exceptions.BlueprintIllegalArgumentException;
import com.atlassian.confluence.plugins.createcontent.services.PromotedTemplateService;
import com.atlassian.confluence.plugins.createcontent.services.TemplateUpdater;
import com.atlassian.confluence.spaces.Space;
import com.atlassian.event.api.EventListener;
import com.atlassian.event.api.EventPublisher;
import com.atlassian.plugin.spring.scanner.annotation.imports.ComponentImport;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;

@Component
public class TemplateUpdatedListener {
    private final TemplateUpdater templateUpdater;
    private final PromotedTemplateService promotedTemplateService;
    private final EventPublisher eventPublisher;

    @Autowired
    public TemplateUpdatedListener(
            final TemplateUpdater templateUpdater,
            final PromotedTemplateService promotedTemplateService,
            final @ComponentImport EventPublisher eventPublisher) {
        this.templateUpdater = templateUpdater;
        this.promotedTemplateService = promotedTemplateService;
        this.eventPublisher = eventPublisher;
    }

    @EventListener
    public void onTempateUpdateEvent(TemplateUpdateEvent event) {
        PageTemplate newTemplate = event.getNewTemplate();
        PageTemplate oldTemplate = event.getOldTemplate();

        if (oldTemplate != null && (oldTemplate.getId() == newTemplate.getId()))
            return;   // the old and new templates are already referenced by the Content Blueprint

        // Update the blueprint content template reference in AO
        templateUpdater.updateContentTemplateRef(newTemplate);
    }

    @EventListener
    public void onTemplateRemoveEvent(TemplateRemoveEvent event) throws BlueprintIllegalArgumentException {
        PageTemplate template = event.getTemplate();
        Space space = template.getSpace();

        if (StringUtils.isBlank(template.getPluginKey()) && space != null) // space level user template
        {
            // Un promote template
            promotedTemplateService.demoteTemplate(template.getId(), space.getKey());
        }

        // Update the blueprint content template reference in AO
        templateUpdater.revertContentTemplateRef(template);
    }

    @PreDestroy
    public void destroy() {
        eventPublisher.unregister(this);
    }

    @PostConstruct
    public void afterPropertiesSet() {
        eventPublisher.register(this);
    }
}
