package com.atlassian.confluence.plugins.createcontent.listeners;

import com.atlassian.confluence.event.events.user.UserRemoveEvent;
import com.atlassian.confluence.plugins.createcontent.services.UserStorageService;
import com.atlassian.event.api.EventListener;
import com.atlassian.event.api.EventPublisher;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;

/**
 * Event listener to pick up users being removed and clean up their corresponding entries from feature discovery.
 *
 * @since 9.1.4
 */
@Component
public class RemoveUserListener {

    private static final Logger log = LoggerFactory.getLogger(RemoveUserListener.class);

    private final EventPublisher eventPublisher;

    private final UserStorageService userStorageService;

    @Autowired
    public RemoveUserListener(EventPublisher eventPublisher, UserStorageService userStorageService) {
        this.eventPublisher = eventPublisher;
        this.userStorageService = userStorageService;
    }

    /**
     * Remove the quick-create discovery key in bandana for the removed user.
     *
     * @param event the event fired when a user is removed.
     */
    @EventListener
    public void userDeleted(UserRemoveEvent event) {
        userStorageService.removeKeyForUser("quick-create", event.getUser());
        log.debug("Removed quick-create discovery record for deleted user");
    }

    @PreDestroy
    public void destroy() {
        eventPublisher.unregister(this);
    }

    @PostConstruct
    public void afterPropertiesSet() {
        eventPublisher.register(this);
    }
}
