package com.atlassian.confluence.plugins.createcontent.listeners;

import com.atlassian.confluence.plugins.createcontent.BlueprintConstants;
import com.atlassian.confluence.plugins.createcontent.actions.BlueprintManager;
import com.atlassian.confluence.plugins.createcontent.api.events.BlueprintPageCreateEvent;
import com.atlassian.confluence.plugins.createcontent.extensions.UserBlueprintConfigManager;
import com.atlassian.confluence.user.AuthenticatedUserThreadLocal;
import com.atlassian.confluence.user.ConfluenceUser;
import com.atlassian.confluence.xwork.FlashScope;
import com.atlassian.event.api.EventListener;
import com.atlassian.event.api.EventPublisher;
import com.atlassian.plugin.spring.scanner.annotation.imports.ComponentImport;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;
import java.util.UUID;

/**
 * Listens for a BlueprintPageCreateevent and adds the key to the flash scope if this is the first time this kind of blueprint
 * has been created
 */
@Component
public class FirstBlueprintCreationListener {
    private final UserBlueprintConfigManager userBlueprintConfigManager;
    private final EventPublisher eventPublisher;

    @Autowired
    public FirstBlueprintCreationListener(
            final UserBlueprintConfigManager userBlueprintConfigManager,
            final @ComponentImport EventPublisher eventPublisher) {
        this.userBlueprintConfigManager = userBlueprintConfigManager;
        this.eventPublisher = eventPublisher;
    }

    @EventListener
    public void onBlueprintCreateEvent(BlueprintPageCreateEvent event) {
        ConfluenceUser user = AuthenticatedUserThreadLocal.get();
        if (user == null)
            return;  // don't show anonymous users the Index-page popup every time.

        UUID id = event.getBlueprint().getId();
        if (userBlueprintConfigManager.isFirstBlueprintOfTypeForUser(id, user)) {
            userBlueprintConfigManager.setBlueprintCreatedByUser(id, user);
            FlashScope.put(BlueprintManager.FIRST_BLUEPRINT_FOR_USER, id);
            FlashScope.put(BlueprintConstants.INDEX_DISABLED, event.getBlueprint().isIndexDisabled());
        }
    }

    @PreDestroy
    public void destroy() {
        eventPublisher.unregister(this);
    }

    @PostConstruct
    public void afterPropertiesSet() {
        eventPublisher.register(this);
    }
}
