package com.atlassian.confluence.plugins.createcontent.impl;

import com.atlassian.bandana.BandanaContext;
import com.atlassian.bandana.BandanaManager;
import com.atlassian.confluence.api.service.exceptions.BadRequestException;
import com.atlassian.confluence.plugins.createcontent.services.UserStorageService;
import com.atlassian.confluence.setup.bandana.ConfluenceBandanaContext;
import com.atlassian.confluence.user.AuthenticatedUserThreadLocal;
import com.atlassian.plugin.spring.scanner.annotation.imports.ComponentImport;
import com.atlassian.user.User;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import javax.ws.rs.core.Response;
import java.util.Collections;
import java.util.List;

import static java.util.regex.Matcher.quoteReplacement;

@Component
public class DefaultUserStorageService implements UserStorageService {
    private final BandanaManager bandanaManager;
    private static final BandanaContext GLOBAL_DATA_CONTEXT = new ConfluenceBandanaContext(DefaultUserStorageService.class.getName());

    private final String SEPARATE_CHARACTER = ",";
    private final List<String> KEY_ACCEPTED = Collections.singletonList("quick-create");

    @Autowired
    public DefaultUserStorageService(final @ComponentImport BandanaManager bandanaManager) {
        this.bandanaManager = bandanaManager;
    }

    public boolean isKeyStoredForCurrentUser(final String key) {
        final User user = AuthenticatedUserThreadLocal.get();
        if (user == null || StringUtils.isEmpty(key) || !KEY_ACCEPTED.contains(key)) return false;

        Object users = bandanaManager.getValue(GLOBAL_DATA_CONTEXT, key);
        return users != null && users.toString().contains(user.getName() + SEPARATE_CHARACTER);
    }

    public void storeKeyForCurrentUser(final String key) {
        final User user = AuthenticatedUserThreadLocal.get();
        if (user == null || user.getName() == null || StringUtils.isEmpty(key) || !KEY_ACCEPTED.contains(key))
            throw new BadRequestException(Response.Status.BAD_REQUEST.getReasonPhrase());

        Object users = bandanaManager.getValue(GLOBAL_DATA_CONTEXT, key);
        if (users == null || !users.toString().contains(user.getName() + SEPARATE_CHARACTER)) {
            bandanaManager.setValue(GLOBAL_DATA_CONTEXT, key, (users == null ? "" : users) + user.getName() + SEPARATE_CHARACTER);
        }
    }

    public void removeKeyForUser(final String key, User user) {
        if (key == null || !KEY_ACCEPTED.contains(key)) {
            throw new UnsupportedOperationException("The key supplied is not valid.");
        }
        if (user != null && user.getName() != null && user.getName().length() != 0) {
            Object bandanaRecord = bandanaManager.getValue(GLOBAL_DATA_CONTEXT, key);
            if (bandanaRecord == null) {
                return;
            }
            String tokenToRemove = user.getName() + SEPARATE_CHARACTER;
            String userlistString = bandanaRecord.toString();

            if (userlistString.contains(tokenToRemove)) {
                String amendedUserList = userlistString.replaceFirst(quoteReplacement(tokenToRemove), "");
                if (!amendedUserList.isEmpty()) {
                    bandanaManager.setValue(GLOBAL_DATA_CONTEXT, key, amendedUserList);
                } else {
                    bandanaManager.removeValue(GLOBAL_DATA_CONTEXT, key);
                }
            }
        } else {
            throw new UnsupportedOperationException("The user supplied is not valid.");
        }
    }
}
