package com.atlassian.confluence.plugins.createcontent.impl;

import com.atlassian.annotations.VisibleForTesting;
import com.atlassian.confluence.plugins.createcontent.ContentBlueprintManager;
import com.atlassian.confluence.plugins.createcontent.api.services.ContentBlueprintSanitiserManager;
import com.atlassian.confluence.plugins.createcontent.rest.entities.CreateBlueprintPageRestEntity;
import com.atlassian.confluence.plugins.createcontent.services.model.CreateBlueprintPageEntity;
import com.atlassian.confluence.spaces.Space;
import com.atlassian.confluence.spaces.SpaceManager;
import com.atlassian.confluence.user.ConfluenceUser;
import com.atlassian.confluence.user.UserAccessor;
import com.atlassian.confluence.util.GeneralUtil;
import com.atlassian.plugin.spring.scanner.annotation.imports.ComponentImport;
import com.atlassian.sal.api.user.UserKey;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.util.Arrays;
import java.util.Map;
import java.util.Objects;
import java.util.UUID;

import static com.atlassian.confluence.plugins.createcontent.impl.DefaultContentBlueprintService.VIEW_PERMISSIONS_USERS;
import static java.util.stream.Collectors.joining;
import static org.slf4j.LoggerFactory.getLogger;

/**
 * @since 10.0.0
 */
@Component
public class DefaultContentBlueprintSanitiserManager implements ContentBlueprintSanitiserManager {

    private final UserAccessor userAccessor;
    private final SpaceManager spaceManager;
    private final ContentBlueprintManager contentBlueprintManager;

    private static final String CONTEXT_SPACE_KEY = "spaceKey";
    private static final String CONTEXT_SPACE_ID = "spaceId";
    private static final Logger log = getLogger(DefaultContentBlueprintSanitiserManager.class);

    @Autowired
    public DefaultContentBlueprintSanitiserManager(
            @ComponentImport UserAccessor userAccessor,
            @ComponentImport SpaceManager spaceManager,
            ContentBlueprintManager contentBlueprintManager) {
        this.userAccessor = userAccessor;
        this.spaceManager = spaceManager;
        this.contentBlueprintManager = contentBlueprintManager;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public CreateBlueprintPageEntity sanitise(CreateBlueprintPageEntity entity) {
        return new CreateBlueprintPageRestEntity.Builder(entity)
                .spaceId(entity.getSpaceId() == 0 ? mapSpaceKeyToSpaceId(entity.getSpaceKey()) : entity.getSpaceId())
                .spaceKey("")
                .context(cleanupContext(entity.getContext(), entity.getContentBlueprintId()))
                .viewPermissionsUsers(cleanupUserIdentifiers(entity.getViewPermissionsUsers()))
                .build();
    }

    @Override
    public CreateBlueprintPageEntity unsanitise(CreateBlueprintPageEntity sanitisedEntity) {
        String sanitisedSpaceKey = sanitisedEntity.getSpaceKey();
        String spaceKey = StringUtils.isEmpty(sanitisedSpaceKey) ? mapSpaceIdToSpaceKey(sanitisedEntity.getSpaceId()) : sanitisedSpaceKey;
        return new CreateBlueprintPageRestEntity.Builder(sanitisedEntity)
                .spaceKey(spaceKey)
                .spaceId(sanitisedEntity.getSpaceId())
                .context(unsanitiseContext(sanitisedEntity.getContext()))
                .build();
    }

    @VisibleForTesting
    Map<String, Object> cleanupContext(Map<String, Object> context, String contentBlueprintId) {
        // If viewPermissionsUsers has been set at the entity level, then it'll also be present in the context and should be cleaned as well.
        if (context.containsKey(VIEW_PERMISSIONS_USERS)) {
            Object viewPermissionsUsersObject = context.get(VIEW_PERMISSIONS_USERS);
            if (viewPermissionsUsersObject instanceof String) {
                context.put(VIEW_PERMISSIONS_USERS, cleanupUserIdentifiers((String) viewPermissionsUsersObject));
            }
        }
        if (context.containsKey(CONTEXT_SPACE_KEY)) {
            Object spaceKey = context.get(CONTEXT_SPACE_KEY);
            if (spaceKey instanceof String) {
                context.remove(CONTEXT_SPACE_KEY);
                // Storing a string instead of a Long because Jackson will try and convert a number to an Integer when unmarshalling this JSON.
                context.put(CONTEXT_SPACE_ID, "" + mapSpaceKeyToSpaceId((String) spaceKey));
            } else {
                log.warn("The spaceKey found in the context '{}' was not the appropriate type (String). Skipping...", spaceKey);
            }
        }
        if (contentBlueprintId != null) {
            UUID uuid = UUID.fromString(contentBlueprintId);
            // find the blueprint based on the given ID
            ContentBlueprint blueprint = contentBlueprintManager.getById(uuid);
            if (blueprint != null) {
                log.debug("Found blueprint {}", blueprint.getIndexKey());
                // these blueprints have context entries containing potential PII that need to be cleaned up.
                switch (blueprint.getIndexKey()) {
                    case "daci-decision":
                        context = sanitiseStorageFormatContextEntries(context, "daciDecisionTemplateXML");
                        return sanitiseUserIdentifierContextEntries(context, "driver", "approver");
                    case "health-monitor":
                        context = sanitiseStorageFormatContextEntries(context, "projectMonitorTemplateXML", "leadershipMonitorTemplateXML", "serviceMonitorTemplateXML");
                        return sanitiseUserIdentifierContextEntries(context, "sponsor", "owner");
                    case "requirements":
                        return sanitiseStorageFormatContextEntries(context, "documentOwner");
                    case "retrospective":
                        context = sanitiseStorageFormatContextEntries(context, "participants", "participantList", "participantTable");
                        return sanitiseUserIdentifierContextEntries(context, "retro-participants");
                    case "decisions":
                        context = sanitiseStorageFormatContextEntries(context, "owner", "mentions");
                        return sanitiseUserIdentifierContextEntries(context, "stakeholders");
                    case "meeting-notes":
                        return sanitiseStorageFormatContextEntries(context, "documentOwner");
                    case "kb-how-to-article":
                        return sanitiseStorageFormatContextEntries(context, "contentbylabelMacro", "jiraIssuesMacro");
                    case "kb-troubleshooting-article":
                        return sanitiseStorageFormatContextEntries(context, "contentbylabelMacro", "jiraIssuesMacro");
                    case "project-poster":
                        return sanitiseStorageFormatContextEntries(context, "templateXML");
                    case "experience-canvas":
                        return sanitiseStorageFormatContextEntries(context, "templateXML");
                    default:
                        log.debug("This blueprint was not recognised as needing context cleanup. Skipping...");
                }
            } else {
                log.warn("Unable to identify the source blueprint for this entity. The entity's context will not be sanitised.");
            }
        }
        return context;
    }

    @VisibleForTesting
    Map<String, Object> unsanitiseContext(Map<String, Object> sanitisedContext) {
        if (sanitisedContext.containsKey(CONTEXT_SPACE_ID)) {
            Object spaceIdObject = sanitisedContext.get(CONTEXT_SPACE_ID);
            try {
                if (spaceIdObject instanceof String) {
                    sanitisedContext.remove(CONTEXT_SPACE_ID);
                    sanitisedContext.put(CONTEXT_SPACE_KEY, mapSpaceIdToSpaceKey(Long.parseLong((String)spaceIdObject)));
                } else {
                    log.warn("The spaceId found in the context '{}' was not the appropriate type (String). Skipping...", spaceIdObject);
                }
            } catch (NumberFormatException e) {
                log.error("Could not convert the stored spaceId entry '{}' back into a spaceKey", spaceIdObject);
            }
        }
        return sanitisedContext;
    }

    private Map<String, Object> sanitiseUserIdentifierContextEntries(Map<String, Object> context, String... keys) {
        for (String key : keys) {
            if (context.containsKey(key)) {
                context.put(key, cleanupUserIdentifiers(((String) context.get(key))));
            }
        }
        return context;
    }

    /*
     * CONFSRVDEV-9865: None of our blueprints passing around storage format fragments require these fragments again.
     * Should be safe to remove each of them.
     */
    private Map<String, Object> sanitiseStorageFormatContextEntries(Map<String, Object> context, String... keys) {
        for (String key : keys) {
            if (context.containsKey(key)) {
                context.put(key, "");
            }
        }
        return context;
    }

    /*
     * CONFSRVDEV-9710: Convert any usernames present in a comma-separated list of user identifiers into their equivalent userkeys to avoid storing username PII.
     */
    @VisibleForTesting
    String cleanupUserIdentifiers(String inputString) {
        if (inputString == null || inputString.isEmpty()) {
            return inputString;
        }
        return Arrays.stream(GeneralUtil.splitCommaDelimitedString(inputString))
                .map(this::mapUserIdentifierToUserKey)
                .filter(Objects::nonNull)
                .collect(joining(","));
    }

    private String mapUserIdentifierToUserKey(String userIdentifier) {
        if (userAccessor.getUserByKey(new UserKey(userIdentifier)) != null) {
            return userIdentifier;
        }
        ConfluenceUser potentialUser = userAccessor.getUserByName(userIdentifier);
        if (potentialUser != null) {
            return (potentialUser.getKey().getStringValue());
        }
        return null;
    }

    private long mapSpaceKeyToSpaceId(String spaceKey) {
        Space space = spaceManager.getSpace(spaceKey);
        return space != null ? space.getId() : 0;
    }

    private String mapSpaceIdToSpaceKey(long spaceId) {
        if (spaceId == 0) {
            return "";
        }
        Space space = spaceManager.getSpace(spaceId);
        return space != null ? space.getKey() : "";
    }
}
