package com.atlassian.confluence.plugins.createcontent.extensions;

import com.atlassian.confluence.languages.LocaleManager;
import com.atlassian.confluence.pages.templates.PageTemplate;
import com.atlassian.confluence.plugin.descriptor.web.ConfluenceWebFragmentHelper;
import com.atlassian.confluence.plugin.module.PluginModuleHolder;
import com.atlassian.confluence.util.i18n.I18NBean;
import com.atlassian.confluence.util.i18n.I18NBeanFactory;
import com.atlassian.plugin.ModuleCompleteKey;
import com.atlassian.plugin.Plugin;
import com.atlassian.plugin.PluginParseException;
import com.atlassian.plugin.Resources;
import com.atlassian.plugin.descriptors.AbstractModuleDescriptor;
import com.atlassian.plugin.elements.ResourceLocation;
import com.atlassian.plugin.loaders.LoaderUtils;
import com.atlassian.plugin.module.ModuleFactory;
import com.atlassian.plugin.spring.scanner.annotation.imports.ComponentImport;
import com.atlassian.plugin.web.ContextProvider;
import com.atlassian.plugin.web.NoOpContextProvider;
import com.atlassian.plugin.web.conditions.ConditionLoadingException;
import com.atlassian.sal.api.net.Request;
import com.atlassian.sal.api.net.RequestFactory;
import com.atlassian.sal.api.net.Response;
import com.atlassian.sal.api.net.ResponseException;
import com.atlassian.sal.api.net.ResponseHandler;
import org.apache.commons.io.IOUtils;
import org.apache.commons.io.input.BoundedInputStream;
import org.dom4j.Element;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.annotation.Nonnull;
import java.io.IOException;
import java.io.InputStream;
import java.io.PrintWriter;
import java.io.StringWriter;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.Map;

import static com.atlassian.confluence.core.BodyType.XHTML;
import static org.apache.commons.lang3.StringUtils.isBlank;
import static org.apache.commons.lang3.StringUtils.isNotBlank;

/**
 * Encapsulates the definition of a content template (one used to create Confluence pages and content).
 *
 * @since 5.0
 */
public class ContentTemplateModuleDescriptor extends AbstractModuleDescriptor<PageTemplate> {
    /**
     * The maximum size of a remote template in bytes.
     */
    private static final int MAX_TEMPLATE_SIZE = 10240;

    private static final Logger log = LoggerFactory.getLogger(ContentTemplateModuleDescriptor.class);

    private final I18NBeanFactory i18NBeanFactory;
    private final LocaleManager localeManager;
    private final RequestFactory<?> requestFactory;

    private ContextProvider contextProvider;
    private ContextProviderConfig contextProviderConfig;
    private URL templateLocator;
    private ModuleCompleteKey moduleCompleteKey;
    private String nameKey;
    private PluginModuleHolder<PageTemplate> pluginModuleHolder;

    /**
     * Special Response Handler that will only read the response up to the desired size.  This is to
     * protect Confluence form blueprints that are too big.
     *
     * @param <T>
     */
    private static class ContentTemplateResponeHandler<T extends Response> implements ResponseHandler<T> {
        private String body;

        @Override
        public void handle(T response) throws ResponseException {
            try {
                InputStream stream = response.getResponseBodyAsStream();
                try {
                    body = IOUtils.toString(new BoundedInputStream(stream, MAX_TEMPLATE_SIZE));
                    if (stream.available() > 0)
                        throw new ResponseException("Template too big (size>" + MAX_TEMPLATE_SIZE + ")");
                } finally {
                    IOUtils.closeQuietly(stream);
                }
            } catch (IOException e) {
                throw new ResponseException(e);
            }
        }

        public String getBody() {
            return body;
        }
    }

    public ContentTemplateModuleDescriptor(
            final @ComponentImport ModuleFactory moduleFactory,
            final @ComponentImport I18NBeanFactory i18NBeanFactory,
            final @ComponentImport LocaleManager localeManager,
            final @ComponentImport RequestFactory<?> requestFactory) {
        super(moduleFactory);

        this.i18NBeanFactory = i18NBeanFactory;
        this.localeManager = localeManager;
        this.requestFactory = requestFactory;
    }

    @Override
    public void init(@Nonnull Plugin plugin, @Nonnull Element element) throws PluginParseException {
        super.init(plugin, element);

        if (isBlank(getKey()))
            throw new PluginParseException("key is a required attribute of <content-template>.");
        if (isBlank(getI18nNameKey())) {
            // CONFDEV-18602 Should throw exception once our Blueprint plugins are updated
            log.warn("i18n-name-key is a required attribute of <content-template> for module: " + getCompleteKey());
//            throw new PluginParseException("i18n-name-key is a required attribute of <content-template>.");
        }

        nameKey = isBlank(getI18nNameKey()) ? "create.content.plugin.plugin.default-template-name" : getI18nNameKey();

        contextProviderConfig = getContextProviderConfig(element);
        templateLocator = getTemplateLocator(element);
        moduleCompleteKey = new ModuleCompleteKey(getCompleteKey());

        pluginModuleHolder = PluginModuleHolder.getInstance(() -> {
            PageTemplate result = new PageTemplate();

            result.setBodyType(XHTML);
            result.setContent(getTemplateContent(templateLocator));
            result.setModuleCompleteKey(moduleCompleteKey);

            I18NBean i18nBean = i18NBeanFactory.getI18NBean(localeManager.getSiteDefaultLocale());

            result.setName(i18nBean.getText(nameKey));

            if (isNotBlank(getDescriptionKey()))
                result.setDescription(i18nBean.getText(getDescriptionKey()));

            return result;
        });
    }

    private String getTemplateContent(URL templateLocator) {
        InputStream inputStream = null;
        try {
            if (log.isDebugEnabled())
                log.debug("load remote template for [ {} ] from [ {} ]", nameKey, templateLocator);

            String protocol = templateLocator.getProtocol();
            if ("http".equals(protocol) || "https".equals(protocol)) {
                Request<?, ?> request = requestFactory.createRequest(Request.MethodType.GET, templateLocator.toString());
                try {
                    ContentTemplateResponeHandler responseHandler = new ContentTemplateResponeHandler();
                    request.execute(responseHandler);
                    return responseHandler.getBody();
                } catch (ResponseException e) {
                    if (log.isDebugEnabled())
                        log.debug("could not load template for [ {} ] from [ {} ]: {}", nameKey, templateLocator, e);

                    return error(e);
                }
            } else {
                inputStream = templateLocator.openStream();
                return IOUtils.toString(inputStream, "UTF-8");
            }
        } catch (IOException e) {
            if (log.isDebugEnabled())
                log.debug("could not load template for [ {} ] from [ {} ]: {}", nameKey, templateLocator, e);

            throw new RuntimeException("Error retrieving template data from URL: " + templateLocator);
        } finally {
            IOUtils.closeQuietly(inputStream);
        }
    }

    /**
     * Provides information to the user about any errors that may have been a result of trying to process or collect
     * a template.
     *
     * @param e an associated exception
     * @return the html string
     */
    private String error(Throwable e) {
        StringWriter writer = new StringWriter();
        writer.append("<h1>An error occurred creating content from template</h1><p>Template: ");
        writer.append(getNameKey());
        writer.append("</p>");
        writer.append("<ac:structured-macro ac:name=\"expand\"><ac:rich-text-body><pre>");
        e.printStackTrace(new PrintWriter(writer, true));
        writer.append("</pre></ac:rich-text-body></ac:structured-macro>");
        return writer.toString();
    }

    @Override
    public void enabled() {
        super.enabled();

        contextProvider = getContextProvider(contextProviderConfig);
        pluginModuleHolder.enabled(getModuleClass());
    }

    @Override
    public void disabled() {
        contextProvider = null;
        pluginModuleHolder.disabled();

        super.disabled();
    }

    private ContextProvider getContextProvider(ContextProviderConfig config) throws PluginParseException {
        if (config == null) {
            return new NoOpContextProvider();
        } else {
            try {
                final ContextProvider context = new ConfluenceWebFragmentHelper().loadContextProvider(config.contextProviderClassName, getPlugin());
                context.init(config.contextProviderParams);
                return context;
            } catch (final ClassCastException ex) {
                throw new PluginParseException("Configured context-provider class does not implement the ContextProvider interface", ex);
            } catch (final ConditionLoadingException ex) {
                throw new PluginParseException("Unable to load the module's display conditions: " + ex.getMessage(), ex);
            }
        }
    }

    private URL getTemplateLocator(Element element) {
        final Resources resources = Resources.fromXml(element);

        ResourceLocation templateLocation = resources.getResourceLocation("download", "template");
        if (templateLocation == null) {
            throw new PluginParseException("You must specify a template resource for the <content-template> tag. Add <resource name=\"template\" type=\"download\" location=\"<insert-path-to-your-template>/template.xml\"/> as a child element of <content-template>.");
        }

        String location = templateLocation.getLocation();
        URL templateLocator = getPlugin().getResource(location);
        if (templateLocator == null) {
            try {
                // local resources that were not found above will throw MalformedURLException and fall past this
                templateLocator = new URL(location);

                // only allow remote http locations for templates (no file:// templates)
                String protocol = templateLocator.getProtocol();
                if (!"http".equals(protocol) && !"https".equals(protocol)) {
                    throw new PluginParseException("Invalid protocol for remote template: " + protocol);
                }

            } catch (MalformedURLException e) {
                // this will be the default action if the location is not found by the plugin
                throw new PluginParseException("Could not load template XML at: " + location);
            }
        }

        if (log.isDebugEnabled())
            log.debug("found resource for content-template [ {} ] at [ {} ]", nameKey, templateLocator);

        return templateLocator;
    }

    private ContextProviderConfig getContextProviderConfig(Element element) {
        Element contextProviderElement = element.element("context-provider");
        if (contextProviderElement == null) {
            return null;
        } else {
            final String contextProviderClassName = contextProviderElement.attributeValue("class");
            final Map<String, String> contextProviderParams = LoaderUtils.getParams(contextProviderElement);

            return new ContextProviderConfig(contextProviderClassName, contextProviderParams);
        }
    }

    public ContextProvider getContextProvider() {
        return contextProvider;
    }

    @Override
    public PageTemplate getModule() {
        return pluginModuleHolder.getModule();
    }

    public String getNameKey() {
        return nameKey;
    }

    public URL getTemplateLocator() {
        return templateLocator;
    }

    @Override
    public Class<PageTemplate> getModuleClass() {
        return PageTemplate.class;
    }

    private static class ContextProviderConfig {
        final String contextProviderClassName;
        final Map<String, String> contextProviderParams;

        ContextProviderConfig(String contextProviderClassName, Map<String, String> contextProviderParams) {
            this.contextProviderClassName = contextProviderClassName;
            this.contextProviderParams = contextProviderParams;
        }
    }
}