package com.atlassian.confluence.plugin.functest;

import com.atlassian.confluence.plugin.functest.helper.HelperFactory;
import com.atlassian.confluence.plugin.functest.remote.soap.stub.ConfluenceSoapService;
import com.atlassian.confluence.plugin.functest.remote.soap.stub.ConfluenceSoapServiceServiceLocator;
import junit.framework.Assert;
import net.sourceforge.jwebunit.junit.WebTester;
import org.apache.commons.httpclient.HttpClient;
import org.apache.commons.httpclient.UsernamePasswordCredentials;
import org.apache.commons.httpclient.auth.AuthScope;
import org.apache.commons.httpclient.methods.GetMethod;
import org.apache.commons.lang.StringUtils;
import org.apache.log4j.Logger;
import org.apache.xmlrpc.XmlRpcClient;
import org.apache.xmlrpc.XmlRpcException;

import javax.servlet.http.HttpServletResponse;
import javax.xml.rpc.ServiceException;
import java.io.File;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URL;
import java.rmi.RemoteException;
import java.util.Arrays;
import java.util.Vector;

public class JWebUnitConfluenceWebTester extends WebTester implements ConfluenceWebTester
{
    private static final Logger LOG = Logger.getLogger(JWebUnitConfluenceWebTester.class);

    private final String protocol;

    private final String hostName;

    private final int port;

    private final String contextPath;

    private final String adminUserName;

    private final String adminPassword;

    private String currentUserName;

    private String currentPassword;

    private File confluencePluginJar;

    private String confluencePluginName;

    private File confluenceBackupZip;

    private String licenseString;


    public JWebUnitConfluenceWebTester(String protocol, String hostName, int port, String contextPath, String adminUserName, String adminPassword)
    {
        this.protocol = protocol;
        this.hostName = hostName;
        this.port = port;
        this.contextPath = null != contextPath && contextPath.trim().equals("/") ? "" : contextPath; // Special processing for context paths, because many get confused that no context path is '/'
        this.adminUserName = adminUserName;
        this.adminPassword = adminPassword;

        setCurrentUserName(adminUserName);
        setCurrentPassword(adminPassword);
        getTestContext().setBaseUrl(getBaseUrl());
        setScriptingEnabled(false);
    }

    public String getProtocol()
    {
        return protocol;
    }

    public String getHostName()
    {
        return hostName;
    }

    public int getPort()
    {
        return port;
    }

    public String getContextPath()
    {
        return contextPath;
    }

    public String getBaseUrl()
    {
        StringBuffer baseUrlBuffer = new StringBuffer(getProtocol())
                .append("://").append(getHostName());

        if (80 != getPort())
            baseUrlBuffer.append(':').append(getPort());
        baseUrlBuffer.append(getContextPath());

        return baseUrlBuffer.toString();
    }

    public String getAdminUserName()
    {
        return adminUserName;
    }

    public String getAdminPassword()
    {
        return adminPassword;
    }

    public String getCurrentUserName()
    {
        return currentUserName;
    }

    public void setCurrentUserName(String currentUserName)
    {
        this.currentUserName = currentUserName;
    }

    public String getCurrentPassword()
    {
        return currentPassword;
    }

    public void setCurrentPassword(String currentPassword)
    {
        this.currentPassword = currentPassword;
    }

    public void login(String username, String password)
    {
        setCurrentUserName(username);
        setCurrentPassword(password);
        login();
    }

    public void login()
    {
        gotoPage("/login.action");
        assertTextNotPresent("You are currently logged in");

        setWorkingForm("loginform");
        setTextField("os_username", getCurrentUserName());
        setTextField("os_password", getCurrentPassword());
        submit("login");

        assertLinkPresentWithText("Log Out");
    }

    public void logout()
    {
        gotoPage("/logout.action");
        assertTextPresent("You have been successfully logged out and any automatic logins removed.");
    }

    public XmlRpcClient getXmlRpcClient() throws MalformedURLException
    {
        return new XmlRpcClient(new StringBuffer(getBaseUrl()).append("/rpc/xmlrpc").toString());
    }

    public ConfluenceSoapService getConfluenceSoapService() throws MalformedURLException, ServiceException
    {
        ConfluenceSoapServiceServiceLocator confluenceSoapServiceServiceLocator = new ConfluenceSoapServiceServiceLocator();
        return confluenceSoapServiceServiceLocator.getConfluenceserviceV1(
                new URL(new StringBuffer(getBaseUrl()).append("/rpc/soap-axis/confluenceservice-v1?wsdl").toString())
        );
    }

    public String loginToXmlRpcService(final String userName, final String password) throws XmlRpcException, IOException
    {
        final XmlRpcClient xmlRpcClient = getXmlRpcClient();

        return (String) xmlRpcClient.execute("confluence1.login",
                new Vector(
                        Arrays.asList(new String[]{userName, password})
                ));
    }

    public void logoutFromXmlRpcService(final String authenticationToken)
    {
        final XmlRpcClient xmlRpcClient;

        if (StringUtils.isNotBlank(authenticationToken))
        {
            try
            {
                xmlRpcClient = getXmlRpcClient();
                xmlRpcClient.execute("confluence1.logout",
                        new Vector(
                                Arrays.asList(
                                        new String[]{authenticationToken}
                                )
                        ));
            }
            catch (final MalformedURLException mUrlE)
            {
                LOG.error("Invalid RPC URL specified.", mUrlE);
            }
            catch (final XmlRpcException xmlRpcE)
            {
                LOG.error("Service request denied.", xmlRpcE);
            }
            catch (final RemoteException re)
            {
                LOG.error("There's an error in Confluence.", re);
            }
            catch (final IOException ioe)
            {
                LOG.error("Can't talk to Confluence.", ioe);
            }
        }
    }

    public String loginToXmlRPcService() throws XmlRpcException, IOException
    {
        return loginToXmlRpcService(getCurrentUserName(), getCurrentPassword());
    }

    public String loginToSoapService(final String userName, final String password)
            throws MalformedURLException, ServiceException, RemoteException
    {
        final ConfluenceSoapService confluenceSoapService = getConfluenceSoapService();

        return StringUtils.isNotBlank(userName) ? confluenceSoapService.login(userName, password) : null;
    }

    public String loginToSoapService() throws MalformedURLException, ServiceException, RemoteException
    {
        return loginToSoapService(getCurrentUserName(), getCurrentPassword());
    }

    public void logoutFromSoapService(final String authenticationToken)
    {
        try
        {
            if (null != authenticationToken)
                getConfluenceSoapService().logout(authenticationToken);

        }
        catch (final MalformedURLException mUrlE)
        {
            LOG.error("Invalid RPC URL specified.", mUrlE);
        }
        catch (final ServiceException se)
        {
            LOG.error("Service request denied.", se);
        }
        catch (final RemoteException re)
        {
            LOG.error("There's an error in Confluence.", re);
        }
        catch (final IOException ioe)
        {
            LOG.error("Can't talk to Confluence.", ioe);
        }
    }

    public void updateLicense() throws IOException
    {
        final String licenseString = getLicenseString();

        if (StringUtils.isNotBlank(licenseString))
        {
            gotoPage("/admin/console.action");
            clickLinkWithText("License Details");

            setWorkingForm("updateLicenseForm");
            setTextField("licenseString", licenseString);

            submit("update");
            assertTextNotPresent("License string is too short");
            assertTextNotPresent("License was not valid");
        }
    }

    public void refreshLicense()
    {
        gotoPage("/admin/refreshlicensing.action");
    }

    public void installPlugin()
    {
        final File confluencePluginJar = getConfluencePluginJar();

        if (null != confluencePluginJar && confluencePluginJar.isFile())
        {
            gotoPage("/admin/console.action");
            clickLinkWithText("Plugins");
            assertTextPresent("Manage Plugins");

            setWorkingForm("plugin-upload");

            setTextField("file_0", getConfluencePluginJar().getAbsolutePath());
            submit("confirm");

            assertLinkPresentWithText(getConfluencePluginName());
        }
    }

    public void restoreData()
    {
        final File confluenceBackupZip = getConfluenceBackupZip();

        if (null != confluenceBackupZip && confluenceBackupZip.isFile())
        {
            gotoPage("/admin/backup.action?synchronous=true");
            assertTextPresent("Restore Confluence Data");

            setWorkingForm(2);
            setTextField("file", confluenceBackupZip.getAbsolutePath());
            submit();

            HelperFactory.createIndexHelper(this).update();
        }
    }

    public void flushCaches()
    {
        gotoPage("/admin/console.action");
        clickLinkWithText("Cache Statistics");
        clickLinkWithText("Flush all");
    }

    public File getConfluencePluginJar()
    {
        return confluencePluginJar;
    }

    public void setConfluencePluginJar(File confluencePluginJar)
    {
        this.confluencePluginJar = confluencePluginJar;
    }

    public String getConfluencePluginName()
    {
        return confluencePluginName;
    }

    public void setConfluencePluginName(String confluencePluginName)
    {
        this.confluencePluginName = confluencePluginName;
    }

    public File getConfluenceBackupZip()
    {
        return confluenceBackupZip;
    }

    public void setConfluenceBackupZip(File confluenceBackupZip)
    {
        this.confluenceBackupZip = confluenceBackupZip;
    }

    public String getLicenseString()
    {
        return licenseString;
    }

    public void setLicenseString(String licenseString)
    {
        this.licenseString = licenseString;
    }

    private String assertXsrfStatusCode(String resourcePath)
    {
        StringBuffer urlWithOsAuthTypeReplaced = new StringBuffer(getBaseUrl())
                .append(
                        resourcePath.replaceAll(
                                "(.*[?&])(os_authType=\\w*)(.*)",
                                "$1os_authType=basic$3"
                        )
                );

        if (urlWithOsAuthTypeReplaced.indexOf("os_authType") < 0)
            urlWithOsAuthTypeReplaced.append(urlWithOsAuthTypeReplaced.indexOf("?") < 0 ? '?' : '&')
                    .append("os_authType=basic");

        HttpClient client = new HttpClient();
        GetMethod httpGet = new GetMethod(urlWithOsAuthTypeReplaced.toString());

        try
        {
            client.getState().setCredentials(
                    new AuthScope(
                            AuthScope.ANY_HOST,
                            AuthScope.ANY_PORT
                    ),
                    new UsernamePasswordCredentials(
                            getCurrentUserName(),
                            getCurrentPassword()
                    )
            );
            httpGet.setDoAuthentication(true);

            Assert.assertEquals(
                    HttpServletResponse.SC_FORBIDDEN,
                    client.executeMethod(httpGet)
            );

            return httpGet.getResponseBodyAsString();
        }
        catch (IOException ioe)
        {
            LOG.error("Error getting HTTP resource " + urlWithOsAuthTypeReplaced, ioe);
            return null;
        }
        finally
        {
            httpGet.releaseConnection();
        }
    }

    /**
     * Asserts that a resource is XSRF protected.
     *
     * @param resourcePath The resource to test for XSRF protection. It must not contain the XSRF token. Must be absolute (e.g. <tt>/admin/plugins.action</tt>).
     */
    public void assertResourceXsrfProtected(String resourcePath)
    {
        Assert.assertFalse("No body text returned for " + resourcePath, StringUtils.isBlank(assertXsrfStatusCode(resourcePath)));
    }

    public void assertXsrfTokenNotPresentFailure(String resourcePath)
    {
        Assert.assertTrue(
                assertXsrfStatusCode(resourcePath).indexOf("a required security token was not present") >= 0
        );
    }

    public void assertXsrfTokenNotValidFailure(String resourcePath)
    {
        Assert.assertTrue(
                assertXsrfStatusCode(resourcePath).indexOf("Your session has expired. You may need to re-submit the form or reload the page.") >= 0
        );
    }
}
