package com.atlassian.confluence.ext.usage.query;

import com.atlassian.confluence.ext.usage.UsageConstants;
import com.atlassian.confluence.ext.usage.UsagePeriod;
import com.atlassian.confluence.spaces.Space;
import org.apache.lucene.document.DateTools;
import org.apache.lucene.index.Term;
import org.apache.lucene.search.BooleanClause;
import org.apache.lucene.search.BooleanQuery;
import org.apache.lucene.search.MatchAllDocsQuery;
import org.apache.lucene.search.Query;
import org.apache.lucene.search.TermQuery;
import org.apache.lucene.search.TermRangeQuery;
import org.apache.lucene.util.BytesRef;

import java.util.Collection;
import java.util.Date;
import java.util.Iterator;

/**
 * A very simple usage query object
 */
public class UsageQuery {
    private String period = UsagePeriod.DAILY.name().toLowerCase();
    private Collection contentTypes;
    private Collection spaces;
    private Collection eventTypes;
    private String columns;
    private Date[] timespan;

    public UsageQuery() {
    }

    public UsageQuery(UsageQuery q) {
        this.period = q.getPeriod();
        this.contentTypes = q.getContentTypes();
        this.spaces = q.getSpaces();
        this.eventTypes = q.getEventTypes();
        this.columns = q.getColumns();
        this.timespan = q.getTimespan();
    }

    private Date[] getTimespan() {
        return timespan;
    }

    public Query getLuceneQuery() {
        BooleanQuery q = new BooleanQuery();

        if (!contentTypes.isEmpty()) {
            BooleanQuery typesQuery = new BooleanQuery();
            for (Iterator iterator = contentTypes.iterator(); iterator.hasNext(); ) {
                String contentType = (String) iterator.next();
                typesQuery.add(new TermQuery(new Term(UsageConstants.FIELD_ENTITYTYPE, contentType)), BooleanClause.Occur.SHOULD);
            }
            q.add(typesQuery, BooleanClause.Occur.MUST);
        }

        if (!eventTypes.isEmpty()) {
            BooleanQuery eventsQuery = new BooleanQuery();
            for (Iterator iterator = eventTypes.iterator(); iterator.hasNext(); ) {
                String eventType = (String) iterator.next();
                eventsQuery.add(new TermQuery(new Term(UsageConstants.FIELD_EVENTTYPE, eventType)), BooleanClause.Occur.SHOULD);
            }
            q.add(eventsQuery, BooleanClause.Occur.MUST);
        }

        if (!spaces.isEmpty()) {
            BooleanQuery spacesQuery = new BooleanQuery();
            for (Iterator iterator = spaces.iterator(); iterator.hasNext(); ) {
                Space space = (Space) iterator.next();
                spacesQuery.add(new TermQuery(new Term(UsageConstants.FIELD_SPACEID, Long.toString(space.getId()))), BooleanClause.Occur.SHOULD);
            }
            q.add(spacesQuery, BooleanClause.Occur.MUST);
        }

        if (timespan != null && timespan.length == 2 && (timespan[0] != null || timespan[1] != null)) {
            Date lower = null;
            Date upper = null;

            if (timespan[0] != null)
                lower = timespan[0];

            if (timespan.length > 0 && timespan[1] != null)
                upper = timespan[1];

            q.add(new TermRangeQuery(UsageConstants.FIELD_DATE,
                    (lower != null ? new BytesRef(DateTools.dateToString(lower, DateTools.Resolution.SECOND)) : null),
                    (upper != null ? new BytesRef(DateTools.dateToString(upper, DateTools.Resolution.SECOND)) : null),
                    true, true), BooleanClause.Occur.MUST);
        }

        if (q.getClauses().length == 0) {
            return new MatchAllDocsQuery();
        }

        return q;
    }

    public String getPeriod() {
        return period;
    }

    public Collection getContentTypes() {
        return contentTypes;
    }

    public Collection getSpaces() {
        return spaces;
    }

    public Collection getEventTypes() {
        return eventTypes;
    }

    public String getColumns() {
        return columns;
    }

    public void setPeriod(String period) {
        this.period = period;
    }

    public void setContentTypes(Collection types) {
        this.contentTypes = types;
    }

    public void setSpaces(Collection spaces) {
        this.spaces = spaces;
    }

    public void setEventTypes(Collection events) {
        this.eventTypes = events;
    }

    public void setColumns(String columns) {
        this.columns = columns;
    }

    public void setTimespan(Date[] timespan) {
        this.timespan = timespan;
    }
}
