package com.atlassian.confluence.ext.usage.macros;

import com.atlassian.bonnie.LuceneException;
import com.atlassian.confluence.content.render.xhtml.ConversionContext;
import com.atlassian.confluence.ext.usage.index.UsageIndexManager;
import com.atlassian.confluence.ext.usage.query.TopUserQuery;
import com.atlassian.confluence.labels.LabelManager;
import com.atlassian.confluence.renderer.radeox.macros.MacroUtils;
import com.atlassian.confluence.spaces.Space;
import com.atlassian.confluence.spaces.SpaceManager;
import com.atlassian.confluence.util.velocity.VelocityUtils;
import com.atlassian.core.util.collection.EasyList;
import com.atlassian.plugin.spring.scanner.annotation.imports.ComponentImport;
import com.atlassian.renderer.v2.RenderUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.util.Collection;
import java.util.Date;
import java.util.List;
import java.util.Map;

public class TopUsersMacro extends AbstractUsageMacro {
    private static final Logger log = LoggerFactory.getLogger(TopUsersMacro.class);

    public TopUsersMacro(
            final UsageIndexManager usageIndexManager,
            @ComponentImport final LabelManager labelManager,
            @ComponentImport final SpaceManager spaceManager) {
        super(usageIndexManager, labelManager, spaceManager);
    }

    @Override
    public String execute(Map<String, String> parameters, String string, ConversionContext conversionContext) {
        final String period = calculatePeriod(parameters);
        final Collection<Space> spaces = calculateSpaces(parameters, conversionContext);
        final Collection display = getDelimitedStringParameter("display", parameters, EasyList.build("title", "count"));
        final Collection<String> contentTypes = calculateTypes(parameters);
        final Collection<String> eventTypes = calculateEvents(parameters);
        final String columns = calculateColumns(parameters);
        final Date[] timespan = calculateTimeSpan(parameters);

        TopUserQuery query = new TopUserQuery();
        query.setPeriod(period);
        query.setSpaces(spaces);
        query.setEventTypes(eventTypes);
        query.setColumns(columns);

        if (timespan != null) {
            query.setTimespan(timespan);
        }

        try {
            List dataset = usageIndexManager.queryTopUsers(query, 10);

            final Map<String, Object> velocityContext = getDefaultMacroVelocityContext();
            velocityContext.put("macro", this);
            velocityContext.put("period", period);
            velocityContext.put("spaces", spaces);
            velocityContext.put("contentTypes", contentTypes);
            velocityContext.put("eventTypes", eventTypes);
            velocityContext.put("display", display);
            velocityContext.put("debug", Boolean.toString("true".equals(parameters.get("debug"))));
            if (timespan != null) {
                velocityContext.put("afterTime", timespan[0]);
                velocityContext.put("beforeTime", timespan[1]);
            }
            velocityContext.put("usageQuery", query);
            velocityContext.put("dataset", dataset);
            return renderTopUsers(velocityContext);
        } catch (LuceneException e) {
            if (log.isErrorEnabled()) {
                log.error("Error querying for data.", e);
            }

            if (e.getCause() instanceof IOException) {
                return RenderUtils.blockError("Error in reading index files to query active users. Please try rebuilding the index.", "");
            }

            return RenderUtils.blockError("Error querying popular content usage: " + e, "");
        } catch (Exception e) {
            if (log.isErrorEnabled())
                log.error("Unable to render macro.", e);

            return RenderUtils.blockError("Error in macro: " + e, "");
        }
    }

    Map<String, Object> getDefaultMacroVelocityContext() {
        return MacroUtils.defaultVelocityContext();
    }

    protected String renderTopUsers(Map velocityContext) {
        return VelocityUtils.getRenderedTemplate("/templates/usage/user-stats.vm", velocityContext);
    }

}
