package com.atlassian.confluence.ext.usage.macros;

import com.atlassian.bonnie.LuceneException;
import com.atlassian.confluence.content.render.xhtml.ConversionContext;
import com.atlassian.confluence.ext.usage.index.UsageIndexManager;
import com.atlassian.confluence.ext.usage.query.ContentUsageQuery;
import com.atlassian.confluence.labels.Label;
import com.atlassian.confluence.labels.LabelManager;
import com.atlassian.confluence.renderer.radeox.macros.MacroUtils;
import com.atlassian.confluence.spaces.Space;
import com.atlassian.confluence.spaces.SpaceManager;
import com.atlassian.confluence.util.velocity.VelocityUtils;
import com.atlassian.core.util.collection.EasyList;
import com.atlassian.plugin.spring.scanner.annotation.imports.ComponentImport;
import com.atlassian.renderer.v2.RenderUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.util.Collection;
import java.util.Date;
import java.util.List;
import java.util.Map;

/**
 * A macro showing the popular content over the last 'N' days.
 * <p>
 * <p>
 * Parameters:
 * <ul>
 * <li>spaces or space - a list of comma-separated space keys, or @all for all spaces. DEFAULT: current space</li>
 * <li>types or type - a list of comma-separated types, or @all for all types. DEFAULT: pages</li>
 * <li>timespan - the timespan over which to calculate popularity. DEFAULT: 1w</li>
 * </ul>
 * <p>
 * Examples: <br>
 * {popular:space=FOO}
 * {popular:types=
 */
public class PopularMacro extends AbstractUsageMacro {
    private static final Logger log = LoggerFactory.getLogger(PopularMacro.class);

    public PopularMacro(
            final UsageIndexManager usageIndexManager,
            @ComponentImport final LabelManager labelManager,
            @ComponentImport final SpaceManager spaceManager) {
        super(usageIndexManager, labelManager, spaceManager);
    }

    @Override
    public String execute(Map<String, String> parameters, String body, ConversionContext conversionContext) {
        final Collection<Space> spaces = calculateSpaces(parameters, conversionContext);
        final Collection<String> contentTypes = calculateTypes(parameters);
        final Collection<Label> labels = calculateLabels(parameters, conversionContext);
        final Collection display = getDelimitedStringParameter("display", parameters, EasyList.build("title", "count"));
        final Date[] timespan = calculateTimeSpan(parameters, "1w");
        // List eventTypes = EasyList.build(UsageConstants.TYPE_VIEW);
        final Collection<String> eventTypes = calculateEvents(parameters);
        int max = getIntParameter("max", parameters, 10);
        String style = getStringParameter("style", parameters, "table");

        ContentUsageQuery query = new ContentUsageQuery();
        query.setSpaces(spaces);
        query.setLabels(labels);
        query.setContentTypes(contentTypes);
        query.setEventTypes(eventTypes);
        if (timespan != null) {
            query.setTimespan(timespan);
        }

        try {
            List popular = usageIndexManager.queryPopular(query, max);

            final Map<String, Object> velocityContext = getDefaultMacroVelocityContext();
            velocityContext.put("macro", this);
            velocityContext.put("spaces", spaces);
            velocityContext.put("contentTypes", contentTypes);
            velocityContext.put("eventTypes", eventTypes);
            velocityContext.put("labels", labels);
            velocityContext.put("debug", Boolean.toString("true".equals(parameters.get("debug"))));
            if (timespan != null) {
                velocityContext.put("afterTime", timespan[0]);
                velocityContext.put("beforeTime", timespan[1]);
            }
            velocityContext.put("usageQuery", query);
            velocityContext.put("popular", popular);
            velocityContext.put("max", max);
            velocityContext.put("style", style);
            velocityContext.put("display", display);
            return renderPopular(velocityContext);
        } catch (LuceneException e) {
            if (log.isErrorEnabled()) {
                log.error("Error querying for data.", e);
            }

            if (e.getCause() instanceof IOException) {
                return RenderUtils.blockError("Error in reading index files to query popular content usage. Please try rebuilding the index.", "");
            }

            return RenderUtils.blockError("Error querying popular content usage: " + e, "");
        } catch (Exception e) {
            if (log.isErrorEnabled()) {
                log.error("Unable to render macro.", e);
            }

            return RenderUtils.blockError("Error in macro: " + e, "");
        }
    }

    Map<String, Object> getDefaultMacroVelocityContext() {
        return MacroUtils.defaultVelocityContext();
    }

    protected String renderPopular(Map velocityContext) {
        return VelocityUtils.getRenderedTemplate("/templates/usage/popular.vm", velocityContext);
    }

}
