package com.atlassian.confluence.ext.usage.macros;

import com.atlassian.confluence.content.render.xhtml.ConversionContext;
import com.atlassian.confluence.content.render.xhtml.DefaultConversionContext;
import com.atlassian.confluence.ext.usage.UsageConstants;
import com.atlassian.confluence.ext.usage.UsagePeriod;
import com.atlassian.confluence.ext.usage.index.UsageIndexManager;
import com.atlassian.confluence.labels.Label;
import com.atlassian.confluence.labels.LabelManager;
import com.atlassian.confluence.macro.Macro;
import com.atlassian.confluence.pages.BlogPost;
import com.atlassian.confluence.pages.Comment;
import com.atlassian.confluence.pages.Page;
import com.atlassian.confluence.renderer.PageContext;
import com.atlassian.confluence.spaces.Space;
import com.atlassian.confluence.spaces.SpaceManager;
import com.atlassian.core.util.DateUtils;
import com.atlassian.core.util.InvalidDurationException;
import com.atlassian.renderer.RenderContext;
import com.atlassian.renderer.v2.RenderMode;
import com.atlassian.renderer.v2.macro.BaseMacro;

import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.StringTokenizer;

/**
 * Base macro used by all usage macros.
 */
public abstract class AbstractUsageMacro extends BaseMacro implements Macro {
    protected final UsageIndexManager usageIndexManager;
    protected final SpaceManager spaceManager;
    protected final LabelManager labelManager;
    public static final String TIMESPAN_FORMAT_STRING = "dd/MM/yy";
    private static final DateFormat timespanDF = new SimpleDateFormat(TIMESPAN_FORMAT_STRING);

    public AbstractUsageMacro(UsageIndexManager usageIndexManager, LabelManager labelManager, SpaceManager spaceManager) {
        this.usageIndexManager = usageIndexManager;
        this.labelManager = labelManager;
        this.spaceManager = spaceManager;
    }

    public boolean isInline() {
        return false;
    }

    public boolean hasBody() {
        return false;
    }

    public RenderMode getBodyRenderMode() {
        return RenderMode.NO_RENDER;
    }

    public BodyType getBodyType() {
        return BodyType.NONE;
    }

    public OutputType getOutputType() {
        return OutputType.BLOCK;
    }

    protected String getEitherParameter(Map parameters, String param1, String param2) {
        String paramValue = (String) parameters.get(param1);

        if (paramValue == null) {
            paramValue = (String) parameters.get(param2);
        }

        return paramValue;
    }

    protected boolean hasEitherParameter(Map parameters, String param1, String param2) {
        return parameters.containsKey(param1) || parameters.containsKey(param2);
    }

    public Space getSpace(String id) {
        return spaceManager.getSpace(Long.parseLong(id));
    }

    public String makeTitleCase(String s) {
        if (s.length() > 0) {
            return Character.toTitleCase(s.charAt(0)) + s.substring(1);
        } else {
            return s;
        }
    }

    protected String calculateColumns(Map parameters) {
        if (parameters.containsKey("columns")) {
            String c = (String) parameters.get("columns");
            if (c == null) {
                return null;
            }

            c = c.toLowerCase();
            if (c.startsWith(UsageConstants.COLUMNS_EVENTS)) {
                return UsageConstants.COLUMNS_EVENTS;
            } else if (c.startsWith(UsageConstants.COLUMNS_SPACES)) {
                return UsageConstants.COLUMNS_SPACES;
            } else if (c.startsWith(UsageConstants.COLUMNS_TYPES)) {
                return UsageConstants.COLUMNS_TYPES;
            }
        }

        return null;
    }

    protected Date[] calculateTimeSpan(Map parameters) {
        return calculateTimeSpan(parameters, "1d");
    }

    protected Date[] calculateTimeSpan(Map parameters, String defaultTimespan) {
        String timespanParam = (String) parameters.get("timespan");

        if (timespanParam == null) {
            timespanParam = defaultTimespan;
        }

        Date[] timespan = new Date[]{null, null};

        String after = null, before = null;

        int seperatorIdx = timespanParam.indexOf(",");
        if (seperatorIdx < 0) {
            after = timespanParam.trim();
            timespan[1] = new Date();
        } else {
            after = timespanParam.substring(0, seperatorIdx);
            before = timespanParam.substring(seperatorIdx + 1).trim();
        }

        if (after != null) {
            timespan[0] = parseDate(after);
        }

        if (before != null) {
            timespan[1] = parseDate(before);
        }

        return timespan;
    }

    private Date parseDate(String span) {
        try {
            long time = DateUtils.getDuration(span) * DateUtils.SECOND_MILLIS;
            return new Date(System.currentTimeMillis() - time);
        } catch (InvalidDurationException e) {
            // must be a specified date
            try {
                return timespanDF.parse(span);
            } catch (ParseException e1) {
                return null;
            }
        }
    }

    protected Collection<String> calculateEvents(Map parameters) {
        final Set<String> events = new HashSet<>();
        if (hasEitherParameter(parameters, "event", "events")) {
            String eventsParam = getEitherParameter(parameters, "event", "events");

            for (StringTokenizer stringTokenizer = new StringTokenizer(eventsParam, ", "); stringTokenizer.hasMoreTokens(); ) {
                String s = stringTokenizer.nextToken().toLowerCase();
                if (s.startsWith(UsageConstants.TYPE_VIEW)) {
                    events.add(UsageConstants.TYPE_VIEW);
                } else if (s.startsWith(UsageConstants.TYPE_CREATE)) {
                    events.add(UsageConstants.TYPE_CREATE);
                } else if (s.startsWith(UsageConstants.TYPE_UPDATE) || s.startsWith("edit")) {
                    events.add(UsageConstants.TYPE_UPDATE);
                } else if (s.startsWith(UsageConstants.TYPE_REMOVE)) {
                    events.add(UsageConstants.TYPE_REMOVE);
                } else if (s.equalsIgnoreCase(UsageConstants.KEYWORD_ALL) || s.equalsIgnoreCase(UsageConstants.KEYWORD_AT_ALL)) {
                    return Collections.emptySet();
                }
            }
        } else { // assume only looking pages
            events.add(UsageConstants.TYPE_VIEW);
        }
        return events;
    }

    protected Collection<String> calculateTypes(Map<String, String> parameters) {
        final Set<String> types = new HashSet<>();
        if (hasEitherParameter(parameters, "type", "types")) {
            String typesParam = getEitherParameter(parameters, "type", "types");

            for (StringTokenizer stringTokenizer = new StringTokenizer(typesParam, ", "); stringTokenizer.hasMoreTokens(); ) {
                String s = stringTokenizer.nextToken().toLowerCase();
                if (s.startsWith(Page.CONTENT_TYPE)) {
                    types.add(Page.CONTENT_TYPE);
                } else if (s.startsWith(Comment.CONTENT_TYPE)) {
                    types.add(Comment.CONTENT_TYPE);
                } else if (s.startsWith(BlogPost.CONTENT_TYPE)) {
                    types.add(BlogPost.CONTENT_TYPE);
                } else if (s.startsWith("news")) {
                    types.add(BlogPost.CONTENT_TYPE);
                } else if (s.startsWith(UsageConstants.SPACE_ENTITY_TYPE)) {
                    types.add(UsageConstants.SPACE_ENTITY_TYPE);
                } else if (s.equalsIgnoreCase(UsageConstants.KEYWORD_ALL) || s.equalsIgnoreCase(UsageConstants.KEYWORD_AT_ALL)) {
                    return Collections.emptySet();
                }
            }
        } else {
            types.add(Page.CONTENT_TYPE);
        }
        return types;
    }

    protected Collection<Space> calculateSpaces(Map<String, String> parameters, ConversionContext conversionContext) {
        // now which spaces are we looking at?
        Set<Space> spaces = new HashSet<Space>();
        if (hasEitherParameter(parameters, "space", "spaces")) {
            String spacesParam = getEitherParameter(parameters, "space", "spaces");

            for (StringTokenizer stringTokenizer = new StringTokenizer(spacesParam, ", "); stringTokenizer.hasMoreTokens(); ) {
                String s = stringTokenizer.nextToken();

                Space space = spaceManager.getSpace(s);
                if (space != null)
                    spaces.add(space);
                else if (s.equalsIgnoreCase(UsageConstants.KEYWORD_ALL) || s.equalsIgnoreCase(UsageConstants.KEYWORD_AT_ALL))
                    return Collections.emptySet();
            }
        } else { // assume only looking at the current space
            PageContext pageContext = conversionContext.getPageContext();

            Space space = null;
            if (pageContext != null)
                space = spaceManager.getSpace(pageContext.getSpaceKey());

            if (space != null)
                spaces.add(space);
        }
        return spaces;
    }

    protected Collection<Label> calculateLabels(Map parameters, ConversionContext conversionContext) {
        // now which spaces are we looking at?
        final Set<Label> labels = new HashSet<>();
        if (hasEitherParameter(parameters, "label", "labels")) {
            String labelsParam = getEitherParameter(parameters, "label", "labels");

            for (StringTokenizer stringTokenizer = new StringTokenizer(labelsParam, ", "); stringTokenizer.hasMoreTokens(); ) {
                String s = stringTokenizer.nextToken();

                Label label = labelManager.getLabel(s);
                if (label != null) {
                    labels.add(label);
                } else {
                    // There is no label for this string, but we need to include it for the search.
                    // We should really just stop here, as there will be no matching results.
                    labels.add(new Label(s));
                }
            }
        }
        return labels;
    }

    protected String calculatePeriod(final Map<String, String> parameters) {
        return UsagePeriod.resolve(parameters.get("period")).name().toLowerCase();
    }

    protected int calculateMax(Map parameters, int defaultMax) {
        return getIntParameter("max", parameters, defaultMax);
    }

    protected int getIntParameter(String paramName, Map parameters, int defaultValue) {
        if (parameters.containsKey(paramName)) {
            String paramValueStr = (String) parameters.get(paramName);

            try {
                return Integer.parseInt(paramValueStr);
            } catch (NumberFormatException e) {
            }
        }

        return defaultValue;
    }

    protected String getStringParameter(String paramName, Map parameters, String defaultValue) {
        if (parameters.containsKey(paramName)) {
            return ((String) parameters.get(paramName)).trim();
        }

        return defaultValue;
    }

    protected Collection getDelimitedStringParameter(String paramName, Map parameters, List defaultValue) {
        if (parameters.containsKey(paramName)) {
            List result = new ArrayList();
            for (StringTokenizer stringTokenizer = new StringTokenizer((String) parameters.get(paramName), ", "); stringTokenizer.hasMoreTokens(); ) {
                String s = stringTokenizer.nextToken();
                result.add(s);
            }
            return result;
        }

        return defaultValue;
    }

    public String execute(Map parameters, String body, RenderContext renderContext) {
        return execute(parameters, body, new DefaultConversionContext(renderContext));
    }

    public abstract String execute(Map<String, String> parameters, String body, ConversionContext conversionContext);
}
