package com.atlassian.confluence.ext.usage.event;

import com.atlassian.confluence.ext.usage.index.UsageIndexManager;
import org.apache.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.util.LinkedList;
import java.util.Queue;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.RejectedExecutionHandler;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;

import static java.util.concurrent.TimeUnit.SECONDS;

@Component
public class UsageTaskQueue {
    private static final int QUEUE_CAPACITY = 1000;
    private static final Logger log = Logger.getLogger(UsageTaskQueue.class);

    private final UsageIndexManager usageIndexManager;
    private final ThreadPoolExecutor executor;

    @Autowired
    public UsageTaskQueue(final UsageIndexManager usageIndexManager) {
        this.usageIndexManager = usageIndexManager;
        this.executor = createExecutor();
    }

    private ThreadPoolExecutor createExecutor() {
        LinkedBlockingQueue<Runnable> queue = new LinkedBlockingQueue<Runnable>(QUEUE_CAPACITY);
        RejectedExecutionHandler handler = new ThrottlingRejectedExecutionHandler(
                (r, executor) -> log.error("Discarding usage event(s) because task queue is full. Usage statistics may not be accurate.")
        );
        return new ThreadPoolExecutor(1, 1, 0, SECONDS, queue, handler);
    }

    public Future<?> addTask(final UsageEventWrapperTask task) {
        return executor.submit(() -> {
            if (log.isDebugEnabled())
                log.debug("UsageEventListener.run: " + task);

            try {
                usageIndexManager.index(task);
            } catch (Throwable t) {
                log.error("Error indexing task: " + task + ": " + t, t);
            }
        });
    }

    /**
     * Waits until the queue of events is completely processed, returning <tt>true</tt>
     * only if all events in the queue were processed within the timeout.
     *
     * @param timeoutMillis the maximum amount of time to wait in milliseconds
     * @return <tt>true</tt> if the queue was completely processed, or false otherwise
     */
    public boolean waitUntilQueueProcessed(long timeoutMillis) {
        // because tasks are processed sequentially, we can wait until a
        // new task is processed
        Future<Boolean> future = executor.submit(() -> true);
        try {
            return future.get(timeoutMillis, TimeUnit.MILLISECONDS);
        } catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            return false;
        } catch (ExecutionException e) {
            throw new RuntimeException(e);
        } catch (TimeoutException e) {
            return false;
        }
    }

    Queue<Runnable> getQueue() {
        return new LinkedList<>(executor.getQueue());
    }
}
