package com.atlassian.confluence.ext.usage.event;

import java.util.concurrent.RejectedExecutionHandler;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;

import static java.util.concurrent.TimeUnit.SECONDS;

/**
 * Provides throttling around another rejected execution handler to call it no more often
 * than a specified frequency. Rejected executions that happen more frequently are discarded.
 */
class ThrottlingRejectedExecutionHandler implements RejectedExecutionHandler {
    private final AtomicLong lastHandled = new AtomicLong(0);
    private final RejectedExecutionHandler handler;
    private final long frequency;
    private final TimeUnit unit;

    /**
     * Throttles the provided handler to being executed at most once per second.
     */
    ThrottlingRejectedExecutionHandler(RejectedExecutionHandler handler) {
        this(handler, 1, SECONDS);
    }

    ThrottlingRejectedExecutionHandler(RejectedExecutionHandler handler, long frequency, TimeUnit unit) {
        this.handler = handler;
        this.frequency = frequency;
        this.unit = unit;
    }

    public void rejectedExecution(Runnable r, ThreadPoolExecutor executor) {
        long now = System.currentTimeMillis();
        long last = lastHandled.get();
        if (now - last <= unit.toMillis(frequency)) {
            // discard executions which occur within our frequency limit
            return;
        }

        // multiple threads can get here, so allow just the one that can successfully
        // update the 'lastHandled' value to continue
        if (lastHandled.compareAndSet(last, now)) {
            handler.rejectedExecution(r, executor);
        }
    }
}
