package com.atlassian.confluence.ext.usage.actions;

import com.atlassian.confluence.ext.usage.macros.AbstractUsageMacro;
import com.atlassian.confluence.spaces.actions.AbstractSpaceAction;
import org.apache.commons.lang3.StringUtils;

import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Date;

/**
 * Base class used by all usage reports.
 * Provides date and period selecting functionality.
 */
public abstract class AbstractUsageReport extends AbstractSpaceAction {
    public static final String PERIOD_DAY = "day";
    public static final String PERIOD_WEEK = "week";
    public static final String PERIOD_MONTH = "month";
    public static final String PARAM_DATE_FORMAT = "yyyyMMdd";
    public static final String CHART_PLUGIN_KEY = "confluence.extra.chart";

    private static final String INTERVAL_HOUR = "Hour";
    private static final String INTERVAL_DAY = "Day";
    private static final String MONTH_TITLE_FORMAT = "MMMM yyyy";
    private static final String DEFAULT_TITLE_FORMAT = "dd MMMM yyyy";

    private Date reportDate = new Date();
    private String period = PERIOD_WEEK;
    private int topItems = 10;

    /**
     * @return Return the number of items to display for popular result lists
     */
    public int getTopItems() {
        return topItems;
    }

    /**
     * Set the number of items to display in popular result lists
     *
     * @param topItems topItems
     */
    public void setTopItems(int topItems) {
        this.topItems = topItems;
    }

    /**
     * Set the date of the report
     *
     * @param date Date in PARAM_DATE_FORMAT
     * @throws ParseException ex
     */
    public void setDate(String date) throws ParseException {
        try {
            setReportDate(getParameterDateFormat().parse(date));
        } catch (ParseException e) {
            // CONF-8124: ignore the failed parsing and continue as if no date was set
        }
    }

    /**
     * Set the report period
     *
     * @param period period
     */
    public void setPeriod(String period) {
        this.period = period;
    }

    /**
     * Format a date for use in a query string
     *
     * @param date date
     * @return Date formatted as PARAM_DATE_FORMAT
     */
    public String formatDateParam(Date date) {
        return getParameterDateFormat().format(date);
    }

    /**
     * @return Date for report
     */
    public Date getReportDate() {
        if (getPeriod().equals(PERIOD_WEEK))
            return getWeekStartDate();

        if (getPeriod().equals(PERIOD_MONTH))
            return getMonthStartDate();

        return reportDate;
    }

    /**
     * @return The next report date based on the current report period
     */
    public Date getNextReportDate() {
        return advanceReportDate(1);
    }

    /**
     * @return The previous report date based on the current report period
     */
    public Date getPreviousReportDate() {
        return advanceReportDate(-1);
    }

    public String getPreviousReportDateParam() {
        return formatDateParam(getPreviousReportDate());
    }

    public String getNextReportDateParam() {
        return formatDateParam(getNextReportDate());
    }

    public void setReportDate(Date reportDate) {
        this.reportDate = reportDate;
    }

    /**
     * @return Report period
     */
    public String getPeriod() {
        return period;
    }

    /**
     * @return Formatted period name
     */
    public String getPeriodName() {
        SimpleDateFormat format = new SimpleDateFormat(getTitleDateFormat());
        Object[] args = new Object[]{format.format(getReportDate())};
        return getText("usage.period." + getPeriod() + ".title", args);
    }

    /**
     * @return Timespan parameter for usage macro
     */
    public String getUsageTimespan() {
        SimpleDateFormat dateFormat = new SimpleDateFormat(AbstractUsageMacro.TIMESPAN_FORMAT_STRING);
        return dateFormat.format(getReportDate()) + "," + dateFormat.format(advanceReportDate(1));
    }

    /**
     * @return True if chart plugin is installed and enabled
     */
    public boolean isChartInstalled() {
        return pluginAccessor.isPluginEnabled(CHART_PLUGIN_KEY);
    }

    /**
     * Render the usage macro as a chart if available
     *
     * @param macro       Usage macro string
     * @param chartParams Parameters to use for charting
     * @return Rendered macro
     */
    public String renderUsageMacro(String macro, String chartParams) {
        return getHelper().renderConfluenceMacro(chartIfAvailable(macro, chartParams));
    }

    /**
     * Wrap a table macro in a chart if the chart plugin is enable; otherwise just render the table macro
     *
     * @param tableMacro  Any macro call that returns a chartable table
     * @param chartParams Parameters to use for charting (passed to the chart macro)
     * @return Rendered macro
     */
    public String chartIfAvailable(String tableMacro, String chartParams) {
        if (!isChartInstalled())
            return tableMacro;

        return "{chart" + (StringUtils.isNotBlank(chartParams) ? (":" + chartParams) : "") + "}" + tableMacro + "{chart}";
    }

    private String getTitleDateFormat() {
        if (PERIOD_MONTH.equals(getPeriod())) {
            return MONTH_TITLE_FORMAT;
        }

        return DEFAULT_TITLE_FORMAT;
    }

    private Date getWeekStartDate() {
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(reportDate);
        calendar.set(Calendar.DAY_OF_WEEK, calendar.getFirstDayOfWeek());
        return calendar.getTime();
    }

    /**
     * Advance the report period by the provided number of perod
     *
     * @param periods Number of periods to advance
     * @return Date of the new period
     */
    private Date advanceReportDate(int periods) {
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(getReportDate());

        String period = getPeriod();

        if (PERIOD_WEEK.equals(period))
            calendar.add(Calendar.DATE, periods * calendar.getMaximum(Calendar.DAY_OF_WEEK));
        else if (PERIOD_MONTH.equals(period))
            calendar.add(Calendar.MONTH, periods);
        else if (PERIOD_DAY.equals(period))
            calendar.add(Calendar.DATE, periods);

        return calendar.getTime();
    }

    private Date getMonthStartDate() {
        Calendar calendar = Calendar.getInstance();
        calendar.setTime(reportDate);
        calendar.set(Calendar.DATE, 1);
        return calendar.getTime();
    }

    private DateFormat getParameterDateFormat() {
        return new SimpleDateFormat(PARAM_DATE_FORMAT);
    }

    /**
     * @return Interval to use for charting
     */
    public String getChartTimeInterval() {
        String period = getPeriod();
        if (PERIOD_DAY.equals(period))
            return INTERVAL_HOUR;

        return INTERVAL_DAY;
    }
}
