/* eslint-disable @atlassian/confluence-server/matching-tests,@atlassian/confluence-server/web-resource-audit */

/**
 * @module confluence/clipboard-helper
 */
define('confluence/clipboard-helper', [
    'document',
    'window',
    'ajs',
    'confluence/api/logger'
], function(
    document,
    window,
    AJS,
    Logger
) {
    'use strict';

    /**
     * Support function to copy text to clipboard if clipboard API fails
     * @param {string} text - text to copy to clipboard
     * @param {function} setTooltipLabel - callback function on after text copied
     * */
    function fallbackCopyToClipboard(text, setTooltipLabel) {
        var textarea = document.createElement('textarea');
        textarea.textContent = text;
        textarea.style.position = 'fixed'; // Prevent scrolling to bottom of page in Microsoft Edge.
        document.body.appendChild(textarea);
        textarea.select();

        try {
            document.execCommand('copy'); // Security exception may be thrown by some browsers.
            setTooltipLabel(AJS.I18n.getText('comment.permalink.description.copied'));
        } catch (err) {
            Logger.error('confluence/clipboard-helper: Failed to copy link (using document.execCommand(\'copy\')).', err);
            setTooltipLabel(AJS.I18n.getText('comment.permalink.description.copied.failed'), err);
        } finally {
            document.body.removeChild(textarea);
        }
    }

    /**
     * Function to copy text to clipboard
     * Based on https://stash.atlassian.com/projects/STASH/repos/stash/browse/plugins/frontend/src/widget/click-to-copy-button/click-to-copy-button.jsx
     *
     * @param {string} text - text to copy to clipboard
     * @param {function} setTooltipLabel - callback function on after text copied
     */
    function copyToClipboard(text, setTooltipLabel) {
        var clipboard = window.isSecureContext && window.navigator && window.navigator.clipboard;
        if (clipboard) {
            clipboard.writeText(text)
                .then(function() {
                    setTooltipLabel(AJS.I18n.getText('comment.permalink.description.copied'));
                })
                .catch(function(err) {
                    Logger.error('confluence/clipboard-helper: Failed to copy link (using clipboard.writeText).', err);
                    setTooltipLabel(AJS.I18n.getText('comment.permalink.description.copied.failed'), err);
                });
        } else {
            fallbackCopyToClipboard(text, setTooltipLabel);
        }
    }
    return {
        copyToClipboard: copyToClipboard
    };
});
