package com.atlassian.confluence.compat.core;

import com.atlassian.confluence.core.PluginDataSourceFactory;
import com.atlassian.fugue.Maybe;

import javax.activation.DataSource;
import javax.annotation.Nullable;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.Optional;

/**
 * Interface com.atlassian.confluence.core.PluginDataSourceFactory was changed
 * This class provides backward compatibility for changed methods
 * @since 1.3.0
 */
public abstract class PluginDataSourceFactoryCompatibilityClass {
    /**
     * In Confluence 7.0.1, the method
     * <pre>
     *      Maybe&lt;Iterable&lt;DataSource&gt;&gt; com.atlassian.confluence.core.PluginDataSourceFactory#resourcesFromModules(String, @Nullable
     *  com.atlassian.plugin.util.collect.Predicate&lt;PluginDataSourceFactory.ResourceView&gt;)
     * </pre>
     * was replaced with
     * <pre>
     *      Optional&lt;Iterable&lt;DataSource&gt;&gt; com.atlassian.confluence.core.PluginDataSourceFactory#getResourcesFromModules(String, @Nullable
     *  java.util.Predicate&lt;PluginDataSourceFactory.ResourceView&gt;)
     * </pre>
     *
     * So this new method was introduced to support plugins which require backward compatibility.
     * It calls the proper method (depending on Confluence version) and works in
     * both Confluence 6.* and 7.*
     *
     * @param pluginDataSourceFactory instance of a class that implements {@code com.atlassian.confluence.core.PluginDataSourceFactory}
     * @param moduleKey module key
     * @param filterByTypeEnumValue name of the filter, for example "IMAGE", "EMBEDDED" etc. See {@code com.atlassian.confluence.core.PluginDataSourceFactory$FilterByType}
     * @return data sources
     */
    public static Iterable<DataSource> getResourcesFromModules(
            PluginDataSourceFactory pluginDataSourceFactory,
            String moduleKey,
            @Nullable String filterByTypeEnumValue) {

        try {
            Class predicateClazz = Class.forName("com.atlassian.confluence.core.PluginDataSourceFactory$FilterByType");
            Object enumConstant = Enum.valueOf(predicateClazz, filterByTypeEnumValue);
            return getResourcesFromModulesMethod(predicateClazz, pluginDataSourceFactory, moduleKey, enumConstant);

        } catch (ClassNotFoundException | NoSuchMethodException | IllegalAccessException | InvocationTargetException e) {
            throw new IllegalStateException(e);
        }
    }

    private static Iterable<DataSource> getResourcesFromModulesMethod(
            Class<?> predicateClazz,
            PluginDataSourceFactory pluginDataSourceFactory,
            String moduleKey,
            Object enumConstant
    ) throws NoSuchMethodException, InvocationTargetException, IllegalAccessException {

        final Class[] callArguments = {
                String.class,
                predicateClazz.getInterfaces()[0]
        };

        try {
            // Confluence 7.* code
            Method method = pluginDataSourceFactory.getClass().getMethod("getResourcesFromModules", callArguments);
            Optional<Iterable<DataSource>> dataSources =
                    (Optional<Iterable<DataSource>>) method.invoke(pluginDataSourceFactory, moduleKey, enumConstant);
            return dataSources.get();

        } catch (NoSuchMethodException e) {
            // Confluence 6.* code
            Method method = pluginDataSourceFactory.getClass().getMethod("resourcesFromModules", callArguments);
            Maybe<Iterable<DataSource>> dataSources =
                    (Maybe<Iterable<DataSource>>) method.invoke(pluginDataSourceFactory, moduleKey, enumConstant);
            return dataSources.get();
        }
    }
}
