package com.atlassian.ccev;

import javax.annotation.ParametersAreNonnullByDefault;
import java.nio.charset.StandardCharsets;
import java.util.Arrays;
import java.util.regex.Pattern;

/**
 * Imitates validator.js's email validation
 *
 * The code is translated to Java from validator.js (MIT license) assuming default options.
 */
@ParametersAreNonnullByDefault
class ValidatorJs {
    private static final Pattern tldBasic = Pattern.compile("^([a-z\\u00a1-\\uffff]{2,}|xn[a-z0-9-]{2,})$", Pattern.CASE_INSENSITIVE);
    private static final Pattern tldSpecialAndSpaces = Pattern.compile("[\\s\\u2002-\\u200B\\u202F\\u205F\\u3000\\uFEFF\\uDB40\\uDC20\\u00A9\\uFFFD]");
    private static final Pattern allNumeric = Pattern.compile("^\\d+$");
    private static final Pattern domainPartBasic = Pattern.compile("^[a-z\\u00a1-\\uffff0-9-]+$", Pattern.CASE_INSENSITIVE);
    private static final Pattern fullWidthChars = Pattern.compile("[\\uff01-\\uff5e]");
    private static final Pattern outerHyphen = Pattern.compile("(^-)|(-$)");
    private static final Pattern quotedEmailUserUtf8 = Pattern.compile("^([\\s\\x01-\\x08\\x0b\\x0c\\x0e-\\x1f\\x7f\\x21\\x23-\\x5b\\x5d-\\x7e\\u00A0-\\uD7FF\\uF900-\\uFDCF\\uFDF0-\\uFFEF]|(\\\\[\\x01-\\x09\\x0b\\x0c\\x0d-\\x7f\\u00A0-\\uD7FF\\uF900-\\uFDCF\\uFDF0-\\uFFEF]))*$", Pattern.CASE_INSENSITIVE);
    private static final Pattern emailUserUtf8Part = Pattern.compile("^[a-z\\d!#$%&'*+\\-/=?^_`{|}~\\u00A0-\\uD7FF\\uF900-\\uFDCF\\uFDF0-\\uFFEF]+$", Pattern.CASE_INSENSITIVE);
    private static final int defaultMaxEmailLength = 254;

    // https://github.com/validatorjs/validator.js/blob/d1a9b6d8c5bd7350d6a7303085f0269f6a99aa9b/src/lib/isEmail.js
    // commit b986f3ddb6e6feea654c649293565e92e73010cc
    static boolean isEmail(String email) {
        if (email.length() > defaultMaxEmailLength) {
            return false;
        }

        final int atIndex = email.lastIndexOf('@');
        if (atIndex < 1) { // disallow emails without '@' and without user
            return false;
        }
        String user = email.substring(0, atIndex);
        String domain = email.substring(atIndex + 1);

        if (user.getBytes(StandardCharsets.UTF_8).length > 64
                || domain.getBytes(StandardCharsets.UTF_8).length > 254
        ) {
            return false;
        }

        if (!isFqdn(domain)) {
            return false;
        }

        if (user.charAt(0) == '"') {
            user = user.substring(1, user.length() -1);
            return quotedEmailUserUtf8.matcher(user).matches();
        }

        final String[] userParts = user.split("\\.", -1);
        return Arrays.stream(userParts).allMatch(p ->
                emailUserUtf8Part.matcher(p).matches()
        );
    }

    /**
     * Checks if given string is a fully qualified domain
     *
     * https://github.com/validatorjs/validator.js/blob/d1a9b6d8c5bd7350d6a7303085f0269f6a99aa9b/src/lib/isFQDN.js
     * commit 3358f45de58727cf47865ec8e32f45175a15d731
     */
    static boolean isFqdn(final String domain) {
        final String[] parts = domain.split("\\.", -1);

        final String tld = parts[parts.length - 1];

        if (parts.length < 2
                || !tldBasic.matcher(tld).matches()
                || tldSpecialAndSpaces.matcher(tld).find()
                || allNumeric.matcher(tld).matches()
        ) {
            return false;
        }

        return Arrays.stream(parts).allMatch(part -> part.length() <= 63
                && domainPartBasic.matcher(part).matches()
                && !fullWidthChars.matcher(part).find()
                && !outerHyphen.matcher(part).find());
    }
}
