package com.atlassian.cache.memory.jmx;

import com.atlassian.cache.CacheManager;
import com.atlassian.cache.ManagedCache;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.annotation.Nonnull;
import javax.management.MBeanServer;
import javax.management.MalformedObjectNameException;
import javax.management.ObjectName;
import java.util.Collection;
import java.util.HashSet;
import java.util.Set;

/**
 * Helper for registering / unregistering JMX MBeans for Guava Cache.
 * @since v3.1
 */
public class MemoryCacheMXBeanRegistrar
{
    static final String MBEAN_NAME_PATTERN = "com.google.common.cache:type=CacheStatistics,CacheManager=JIRA Cache Manager,name=%s";

    private final Logger logger = LoggerFactory.getLogger(MemoryCacheMXBeanRegistrar.class);

    private MBeanServer mbeanServer;
    private CacheManager cacheManager;
    private Set<ObjectName> registeredMBeans = new HashSet<>();

    /**
     * Enables collecting JMX metrics.
     * <p>
     * Note: it must be called to allow the method <code>registerMBean</code> registering MBeans.
     *</p>
     *
     * @param mbeanServer {@link MBeanServer} instance used for registering and unregistering MBeans
     * @param cacheManager {@link CacheManager} instance used for retrieving caches metrics
     */
    public void enableCollectingJMXMetrics(@Nonnull MBeanServer mbeanServer, @Nonnull CacheManager cacheManager)
    {
        this.mbeanServer = mbeanServer;
        this.cacheManager = cacheManager;

        Collection<ManagedCache> caches = cacheManager.getManagedCaches();
        for(ManagedCache cache: caches)
        {
            registerMBean(cache.getName());
        }
    }

    /**
     * Registers JMX MBean for the specified cache name.
     * <p>
     * Note: it will not register anything if the method <code>enableCollectingJMXMetrics</code> was not called before
     * or unregisterMBeans was called .
     * </p>
     *
     * @param cacheName the cache name for which MBean should be registered
     */
    public void registerMBean(@Nonnull String cacheName)
    {
        if(mbeanServer != null)
        {
            try
            {
                ObjectName name = buildName(cacheName);

                if (registeredMBeans.contains(name))
                {
                    logger.debug("MBean: " + name.getCanonicalName() + " has been registered previously");
                    return;
                }

                MemoryCacheMXBean mbean = new MemoryCacheMXBeanImpl(cacheManager, cacheName);
                mbeanServer.registerMBean(mbean, name);
                registeredMBeans.add(name);
            }
            catch (Exception e)
            {
                logger.warn("Could not register JMX MBean for Guava Cache: " + cacheName, e);
            }
        }
    }

    /**
     * Unregisters all registered JMX MBeans.
     * <p>
     * Note: it will unregister all registered Cache JMX MBeans
     *</p>
     */
    public void unregisterMBeans()
    {
        if(mbeanServer != null)
        {
            for(ObjectName mbeanName: registeredMBeans)
            {
                try
                {
                    mbeanServer.unregisterMBean(mbeanName);
                }
                catch (Exception e)
                {
                    logger.warn("Could not unregister JMX MBean: " + mbeanName.getCanonicalName(), e);
                }
            }
            registeredMBeans.clear();
            mbeanServer = null;
        }
    }

    private ObjectName buildName(String cacheName) throws MalformedObjectNameException
    {
        return new ObjectName(String.format(MBEAN_NAME_PATTERN, cacheName));
    }
}
