package com.atlassian.cache.memory;

import java.util.Collection;
import java.util.SortedMap;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import com.atlassian.cache.Cache;
import com.atlassian.cache.CacheEntryListener;
import com.atlassian.cache.CacheException;
import com.atlassian.cache.CacheSettings;
import com.atlassian.cache.CacheStatisticsKey;
import com.atlassian.cache.impl.CacheEntryListenerSupport;
import com.atlassian.cache.impl.DefaultCacheEntryListenerSupport;
import com.atlassian.util.concurrent.Supplier;

import com.google.common.base.Throwables;
import com.google.common.cache.LoadingCache;
import com.google.common.cache.RemovalListener;
import com.google.common.cache.RemovalNotification;
import com.google.common.collect.ImmutableSortedMap;
import com.google.common.util.concurrent.UncheckedExecutionException;

import static com.atlassian.cache.memory.DelegatingCacheStatistics.toStatistics;

/**
 * A Cache that delegates Concurrent Map.
 *
 * @since 2.0
 */
class DelegatingCache<K, V> extends ManagedCacheSupport implements Cache<K, V>
{
    private final com.google.common.cache.Cache<K, V> internalCache;
    private final CacheEntryListenerSupport<K, V> listenerSupport;

    private DelegatingCache(final com.google.common.cache.Cache<K, V> internalCache, String name, CacheSettings settings)
    {
        super(name, settings);
        this.internalCache = internalCache;
        this.listenerSupport = new DefaultCacheEntryListenerSupport<K, V>();
    }

    static <K, V> DelegatingCache<K, V> create(final LoadingCache<K, V> internalCache, String name,
            CacheSettings settings, BlockingCacheLoader<K, V> cacheLoader)
    {
        return new DelegatingLoadingCache<K, V>(internalCache, name, settings, cacheLoader);
    }

    static <K, V> DelegatingCache<K, V> create(final com.google.common.cache.Cache<K, V> internalCache, String name, CacheSettings settings)
    {
        return new DelegatingCache<K, V>(internalCache, name, settings);
    }

    @Override
    public boolean containsKey(@Nonnull K key)
    {
        return null != internalCache.getIfPresent(key);
    }

    @Nonnull
    @Override
    public Collection<K> getKeys()
    {
        try
        {
            return internalCache.asMap().keySet();
        }
        catch (Exception e)
        {
            throw new CacheException(e);
        }
    }

    @Override
    public void put(@Nonnull final K key, @Nonnull final V value)
    {
        try
        {
            V oldValue = internalCache.asMap().put(key, value);
            if (oldValue == null)
            {
                // Here we care only for the case when oldValue was null, e.g. missing as in the other
                // cases DelegatingRemovalListener will be called with REPLACED notification
                listenerSupport.notifyAdd(key, value);
            }
        }
        catch (Exception e)
        {
            throw new CacheException(e);
        }
    }

    @Override
    public V get(@Nonnull final K key)
    {
        return internalCache.getIfPresent(key);
    }

    @Nonnull
    @Override
    public V get(@Nonnull final K key, @Nonnull final com.atlassian.cache.Supplier<? extends V> valueLoader)
    {
        try
        {
            return internalCache.get(key, new Callable<V>()
            {
                @Override
                public V call()
                {
                    return valueLoader.get();
                }
            });
        }
        catch (UncheckedExecutionException e)
        {
            Throwable cause = e.getCause();
            Throwables.propagateIfInstanceOf(cause, CacheException.class);
            throw new CacheException(cause);
        }
        catch (ExecutionException e)
        {
            Throwable cause = e.getCause();
            Throwables.propagateIfInstanceOf(cause, CacheException.class);
            throw new CacheException(cause);
        }
        catch (Exception e)
        {
            Throwables.propagateIfInstanceOf(e, CacheException.class);
            throw new CacheException(e);
        }
    }

    @Override
    public void remove(@Nonnull final K key)
    {
        try
        {
            internalCache.invalidate(key);
        }
        catch (Exception e)
        {
            throw new CacheException(e);
        }
    }

    @Override
    public void removeAll()
    {
        try
        {
            internalCache.invalidateAll();
        }
        catch (Exception e)
        {
            throw new CacheException(e);
        }
    }

    @Override
    public V putIfAbsent(@Nonnull K key, @Nonnull V value)
    {
        try
        {
            V oldValue = internalCache.asMap().putIfAbsent(key, value);
            if (oldValue == null)
            {
                // Here we care only for the case when oldValue was null, e.g. missing as in the other
                // cases DelegatingRemovalListener will be called with REPLACED notification
                listenerSupport.notifyAdd(key, value);
            }
            return oldValue;
        }
        catch (Exception e)
        {
            throw new CacheException(e);
        }
    }

    @Override
    public boolean remove(@Nonnull K key, @Nonnull V value)
    {
        try
        {
            return internalCache.asMap().remove(key, value);
        }
        catch (Exception e)
        {
            throw new CacheException(e);
        }
    }

    @Override
    public boolean replace(@Nonnull K key, @Nonnull V oldValue, @Nonnull V newValue)
    {
        try
        {
            return internalCache.asMap().replace(key, oldValue, newValue);
        }
        catch (Exception e)
        {
            throw new CacheException(e);
        }
    }

    @Nonnull
    @Override
    public SortedMap<CacheStatisticsKey, Supplier<Long>> getStatistics()
    {
        if (isStatisticsEnabled())
        {
            return toStatistics(internalCache);
        }
        else
        {
            return ImmutableSortedMap.of();
        }
    }

    @Override
    public void clear()
    {
        removeAll();
    }

    @Override
    public boolean equals(@Nullable final Object other)
    {
        if (other instanceof DelegatingCache)
        {
            DelegatingCache<?, ?> otherDelegatingCache = (DelegatingCache<?, ?>)other;
            if (internalCache.equals(otherDelegatingCache.internalCache))
            {
                return true;
            }
        }
        return false;
    }

    @Override
    public int hashCode()
    {
        return 3 + internalCache.hashCode();
    }

    @Override
    public void addListener(@Nonnull CacheEntryListener<K, V> listener, boolean includeValues)
    {
        listenerSupport.add(listener, includeValues);
    }

    @Override
    public void removeListener(@Nonnull CacheEntryListener<K, V> listener)
    {
        listenerSupport.remove(listener);
    }

    protected static class DelegatingRemovalListener<K, V> implements RemovalListener<K, V>
    {
        private DelegatingCache<K, V> cache;

        protected void onSupply(K key, V value)
        {
            cache.listenerSupport.notifyAdd(key, value);
        }

        @Override
        public void onRemoval(@Nonnull RemovalNotification<K, V> notification)
        {
            switch (notification.getCause())
            {
                case COLLECTED:
                case EXPIRED:
                    cache.listenerSupport.notifyEvict(notification.getKey(), notification.getValue());
                    break;
                case EXPLICIT:
                    cache.listenerSupport.notifyRemove(notification.getKey(), notification.getValue());
                    break;
                case REPLACED:
                    K key = notification.getKey();
                    cache.listenerSupport.notifyUpdate(key, cache.internalCache.getIfPresent(key),
                            notification.getValue());
                    break;
            }
        }

        public void setCache(DelegatingCache<K, V> cache)
        {
            this.cache = cache;
        }
    }

    void rejectNullKey(K key)
    {
        if (key == null)
        {
            throw new CacheException(new NullPointerException("Null keys are not supported"));
        }
    }

    static class DelegatingLoadingCache<K, V> extends DelegatingCache<K, V>
    {
        private final LoadingCache<K, V> internalCache;
        private final BlockingCacheLoader<K, V> cacheLoader;

        private DelegatingLoadingCache(final com.google.common.cache.LoadingCache<K, V> internalCache,
                final String name, final CacheSettings settings, BlockingCacheLoader<K, V> cacheLoader)
        {
            super(internalCache, name, settings);
            this.internalCache = internalCache;
            this.cacheLoader = cacheLoader;
        }

        @Override
        public V get(@Nonnull final K key)
        {
            // Up front check so that postGetCleanup doesn't have to worry about it
            rejectNullKey(key);

            try
            {
                return internalCache.get(key);
            }
            catch (UncheckedExecutionException e)
            {
                Throwable cause = e.getCause();
                Throwables.propagateIfInstanceOf(cause, CacheException.class);
                throw new CacheException(cause);
            }
            catch (ExecutionException e)
            {
                Throwable cause = e.getCause();
                Throwables.propagateIfInstanceOf(cause, CacheException.class);
                throw new CacheException(cause);
            }
            catch (Exception e)
            {
                Throwables.propagateIfInstanceOf(e, CacheException.class);
                throw new CacheException(e);
            }
            finally
            {
                cacheLoader.postGetCleanup(key);
            }
        }

        @Override
        public void remove(@Nonnull K key)
        {
            cacheLoader.acquire(key);
            try
            {
                super.remove(key);
            }
            finally
            {
                cacheLoader.release(key);
            }
        }



        @Override
        public void removeAll()
        {
            cacheLoader.removeAllLock().lock();
            try
            {
                super.removeAll();
            }
            finally
            {
                cacheLoader.removeAllLock().unlock();
            }
        }
    }
}

