package com.atlassian.cache.memory;

import javax.annotation.Nonnull;

import com.atlassian.cache.CacheLoader;
import com.atlassian.cache.CacheSettings;
import com.atlassian.cache.CacheSettingsBuilder;
import com.atlassian.cache.CachedReference;
import com.atlassian.cache.ManagedCache;
import com.atlassian.cache.impl.AbstractCacheManager;
import com.atlassian.cache.impl.ReferenceKey;
import com.atlassian.cache.impl.StrongSupplier;
import com.atlassian.cache.impl.WeakSupplier;
import com.atlassian.util.concurrent.Supplier;

import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.LoadingCache;

import static java.util.concurrent.TimeUnit.MILLISECONDS;

/**
 * Maintains a mapping of name -> Cache and provides factory methods for creating and getting caches.
 *
 * @since 2.0
 */
public class MemoryCacheManager extends AbstractCacheManager
{
    @Nonnull
    public MemoryCacheManager()
    {
        super(null);
    }

    @Nonnull
    @SuppressWarnings("unchecked")
    @Override
    public <V> CachedReference<V> getCachedReference(@Nonnull final String name,
                                                     @Nonnull final com.atlassian.cache.Supplier<V> supplier,
                                                     @Nonnull final CacheSettings settings)
    {
        // Force the cache settings to be flushable and a maximum size of one.
        final CacheSettings overridenSettings = settings.override(
                new CacheSettingsBuilder().flushable().maxEntries(1).build());


        return (DelegatingCachedReference<V>) cacheCreationLocks.get(name).withLock(new com.atlassian.util.concurrent.Supplier<ManagedCache>()
            {
            @Override
            public ManagedCache get()
            {
                // We need to always create a new instance as any old loader may belong to a plugin that has gone away,
                // resulting in a whole world of ClassLoader pain.
                final DelegatingCachedReference.DelegatingReferenceRemovalListener<V> listener = new DelegatingCachedReference.DelegatingReferenceRemovalListener<V>();

                LoadingCache<ReferenceKey, V> computingCache = createCacheBuilder(overridenSettings)
                        .removalListener(listener)
                        .build(new com.google.common.cache.CacheLoader<ReferenceKey, V>()
                        {
                            @Override
                            public V load(final ReferenceKey key) throws Exception
                            {
                                V value = supplier.get();
                                listener.onSupply(value);
                                return value;
                            }
                        });

                DelegatingCachedReference cache = DelegatingCachedReference.create(computingCache, name, overridenSettings);
                listener.setCachedReference(cache);
                caches.put(name, new WeakSupplier<ManagedCache>(cache));
                return cache;
            }
        });
    }

    @SuppressWarnings("unchecked")
    protected ManagedCache createSimpleCache(@Nonnull final String name, @Nonnull final CacheSettings settings)
    {

        Supplier<ManagedCache> cacheSupplier = caches.get(name);
        if (cacheSupplier != null)
        {
            ManagedCache cache = cacheSupplier.get();
            if (cache != null)
            {
                return cache;
            }
        }
        return cacheCreationLocks.get(name).withLock(new com.atlassian.util.concurrent.Supplier<ManagedCache>()
            {
                @Override
                public ManagedCache get()
                {
                    if (!caches.containsKey(name))
                    {
                        DelegatingCache.DelegatingRemovalListener listener = new DelegatingCache.DelegatingRemovalListener();

                        final Cache<Object, Object> simpleCache = createCacheBuilder(settings)
                                .removalListener(listener)
                                .build();

                        DelegatingCache cache = DelegatingCache.create(simpleCache, name, settings);
                        listener.setCache(cache);

                        caches.put(name, new StrongSupplier<ManagedCache>(cache));
                    }
                    return caches.get(name).get();
                }
            });
    }

    @SuppressWarnings("unchecked")
    protected <K, V> ManagedCache createComputingCache(@Nonnull final String name, @Nonnull final CacheSettings settings, final CacheLoader<K, V> loader)
    {
        return cacheCreationLocks.get(name).withLock(new com.atlassian.util.concurrent.Supplier<ManagedCache>()
        {
            @Override
            public ManagedCache get()
            {
                // We need to always create a new instance as any old loader may belong to a plugin that has gone away,
                // resulting in a whole world of ClassLoader pain.
                final DelegatingCache.DelegatingRemovalListener listener = new DelegatingCache.DelegatingRemovalListener();

                LoadingCache<K, V> computingCache = createCacheBuilder(settings)
                        .removalListener(listener)
                        .build(new com.google.common.cache.CacheLoader<K, V>()
                        {
                            @Override
                            public V load(final K key) throws Exception
                            {
                                V value = loader.load(key);
                                listener.onSupply(key, value);
                                return value;
                            }
                        });

                DelegatingCache cache = DelegatingCache.create(computingCache, name, settings);
                listener.setCache(cache);
                caches.put(name, new WeakSupplier<ManagedCache>(cache));
                return cache;
            }
        });
    }

    private CacheBuilder createCacheBuilder(CacheSettings settings)
    {
        final CacheBuilder cacheBuilder = CacheBuilder.newBuilder();
        if (null != settings.getMaxEntries())
        {
            cacheBuilder.maximumSize(settings.getMaxEntries());
        }

        if (null != settings.getExpireAfterAccess())
        {
            cacheBuilder.expireAfterAccess(settings.getExpireAfterAccess(), MILLISECONDS);
        }
        else if (null != settings.getExpireAfterWrite())
        {
            cacheBuilder.expireAfterWrite(settings.getExpireAfterWrite(), MILLISECONDS);
        }
        return cacheBuilder;
    }
}
